import React, { useCallback, useContext } from 'react';

import { classNames } from '@yandex-infracloud-ui/libs';
import { Link, Spin } from '@yandex-cloud/uikit';

import { TStage } from '../../../../../../proto-typings';
import {
   HistoryRequestKeys,
   getHistoryEventId,
   HistoryEvent,
   getHistoryEventStageRevision,
} from '../../../../../../models/ui';
import { useNetworkErrors } from '../../../../../../redux';
import { YpErrorTooltip } from '../../../../../../components/network';
import { StageHistoryListRow } from './StageHistoryListRow';
import { useFetchHistory } from '../../hooks';
import { HistoryListLoadedContext } from '../../context';

import classes from './StageHistoryList.module.css';

interface Props {
   stage: TStage;
   historyList: HistoryEvent[];
}

export const StageHistoryList: React.FC<Props> = React.memo(({ stage, historyList }) => {
   const requestKey = HistoryRequestKeys.getOne({ id: stage.meta!.id });
   const networkErrors = useNetworkErrors([requestKey]);
   const historyListError = networkErrors[requestKey];

   const lastEvent = historyList[historyList.length - 1];
   const isFullHistory = lastEvent && getHistoryEventStageRevision(lastEvent) === 1;

   const { fetchHistoryParts, isLoaded } = useFetchHistory(stage.meta!.id);

   const isGlobalLoaded = useContext(HistoryListLoadedContext);

   const onShowMoreClick = useCallback(() => {
      fetchHistoryParts();
   }, [fetchHistoryParts]);

   return (
      <div data-test={'StageHistory:List'}>
         {historyList && historyList.length ? (
            <>
               <table
                  className={classNames(
                     classes.historyTable,
                     classes.historyTableLarge,
                     classes.historyTableSelectable,
                  )}
               >
                  <thead>
                     <tr>
                        <th>Revision</th>
                        <th>Created</th>
                        <th>Owner</th>
                        <th>Comment</th>
                        <th aria-label={'Actions'} />
                     </tr>
                  </thead>
                  <tbody>
                     {historyList.map(historyEvent => (
                        <StageHistoryListRow
                           key={`history-${getHistoryEventId(historyEvent)}`}
                           historyEvent={historyEvent}
                           stageId={stage.meta!.id}
                           currentRevision={stage.spec!.revision}
                        />
                     ))}
                  </tbody>
               </table>
               {!isLoaded || !isGlobalLoaded ? (
                  <div
                     className={classNames({
                        [classes.spinner]: !historyListError?.error,
                        [classes.spinnerWithError]: historyListError?.error,
                     })}
                  >
                     <Spin size={'xs'} />
                  </div>
               ) : (
                  !isFullHistory && (
                     <div
                        className={classNames({
                           [classes.loadAction]: !historyListError?.error,
                           [classes.loadActionWithError]: historyListError?.error,
                        })}
                     >
                        <Link onClick={onShowMoreClick}>Show more</Link>
                     </div>
                  )
               )}
            </>
         ) : null}
         {historyListError?.error && (
            <div className={classes.error}>
               <YpErrorTooltip error={historyListError.error} request={historyListError.request} />
            </div>
         )}
      </div>
   );
});

StageHistoryList.displayName = 'StageHistoryList';
