import { Button } from '@yandex-cloud/uikit';
import { YCSelect } from '@yandex-data-ui/common';
import React from 'react';
import { YamlEditor } from '@yandex-infracloud-ui/monaco-editor';

import { getRevisionItem } from '../../model';
import { getHistoryEventStageRevision, HistoryEvent } from '../../../../../../models/ui';
import { useHistoryEventText, useHistoryGoto } from '../../hooks';
import { useJSONSchemaByPath } from '../../../../../../json-schema';

import { StageHistoryEventActions } from '../StageHistoryEventActions/StageHistoryEventActions';
import { StageHistoryEventDate } from '../StageHistoryEventDate/StageHistoryEventDate';
import { StageHistoryMessageBlock } from '../StageHistoryMessageBlock/StageHistoryMessageBlock';
import { StageHistoryUserLink } from '../StageHistoryUserLink/StageHistoryUserLink';

import classes from './StageHistorySpecSingle.module.css';

interface Props {
   stageId: string;
   currentRevision: number;
   revisionId: number;
   availableRevisions: number[];
   historyEvent: HistoryEvent;
}

export const StageHistorySpecSingle: React.FC<Props> = React.memo(
   ({ revisionId, availableRevisions, historyEvent, stageId, currentRevision }) => {
      const { gotoHistoryEvent, gotoHistoryCompare } = useHistoryGoto(stageId);
      const { historyEventText } = useHistoryEventText(stageId, revisionId);
      useJSONSchemaByPath('/yp/TStage');

      return (
         <>
            <div style={{ display: 'grid', gridTemplateColumns: '1fr auto', padding: '8px 0' }}>
               <div className={classes.spec} data-test={'StageHistory:RevisionSpec'}>
                  <div className={'revision-1'} data-test={'StageHistory:SelectRevision'}>
                     <YCSelect
                        value={String(revisionId)}
                        items={availableRevisions.map(getRevisionItem)}
                        onUpdate={(value: string) => gotoHistoryEvent(Number(value))}
                     />
                  </div>
                  {availableRevisions.length > 1 && (
                     <div className={classes.compareRevisions} data-test={'StageHistory:CompareRevisions'}>
                        <Button
                           view={'flat-info'}
                           onClick={event => {
                              event.stopPropagation();
                              gotoHistoryCompare(
                                 revisionId,
                                 Number(availableRevisions.find(rId => rId !== revisionId)),
                                 true,
                              );
                           }}
                        >
                           Compare revisions
                        </Button>
                     </div>
                  )}
               </div>

               <div data-test={'StageHistory:DiffApply'}>
                  {currentRevision === getHistoryEventStageRevision(historyEvent) ? (
                     <div className={classes.currentRevision} data-test={'StageHistory:CurrentRevision'}>
                        Current revision
                     </div>
                  ) : (
                     <StageHistoryEventActions historyEvent={historyEvent} stageId={stageId} />
                  )}
               </div>
            </div>

            <div className={classes.revisionInfo}>
               <StageHistoryMessageBlock historyEvent={historyEvent} />
               <div className={classes.metaInfo}>
                  Created on <StageHistoryEventDate historyEvent={historyEvent} /> by{' '}
                  <StageHistoryUserLink historyEvent={historyEvent} />
               </div>
            </div>

            <div style={{ height: '100%' }}>
               <YamlEditor
                  value={historyEventText}
                  isAutoMaxHeight={true}
                  minRows={30}
                  uri={'https://deploy.yandex-team.ru/yp/TStage'}
               />
            </div>
         </>
      );
   },
);

StageHistorySpecSingle.displayName = 'StageHistorySpecSingle';
