import React, { useCallback, useMemo } from 'react';

import { faSpinner, faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';

import { EditSharedNode } from '../../../../../../components/lib';
import { StageHugeForm } from '../../../../../../components/stage-huge-form';
import { StageHugeFormMode, StageHugeFormShowPage } from '../../../../../../components/stage-huge-form/models';
import { useFormStage } from '../../../../../../redux';
import { RenderArgs } from '../../../../../../components/stage-huge-form/StageHugeForm';
import { useHistoryGoto } from '../../hooks';
import { HistoryEvent } from '../../../../../../models/ui';

interface Props {
   asIs: boolean;
   revisionId: number;
   historyEvent: HistoryEvent;
   stageId: string;
}

export const StageRollbackInternalForm: React.FC<Props> = ({ asIs, revisionId, historyEvent, stageId }) => {
   const formMode = asIs ? StageHugeFormMode.ApplyAsIs : StageHugeFormMode.Apply;
   const { rawStage: latestRawStage } = useFormStage(formMode, stageId);
   const { gotoHistoryEvent } = useHistoryGoto(stageId);

   const rawStage = useMemo(
      () =>
         latestRawStage
            ? {
                 ...latestRawStage,

                 spec: {
                    ...historyEvent.result.spec,
                    revision: latestRawStage.spec?.revision,
                 },
              }
            : null,
      [latestRawStage, historyEvent.result],
   );

   const renderChildren = useCallback(
      (form: RenderArgs) => (
         <>
            <EditSharedNode replace={true}>
               {!asIs && form.showPage === StageHugeFormShowPage.Edit && form.hasUnsavedChanges ? (
                  <Button onClick={form.revert} extraProps={{ 'data-e2e': 'StageHugeForm:RevertUnsavedChangesButton' }}>
                     <FontAwesomeIcon icon={faUndo} />
                     &nbsp;&nbsp; Revert changes
                  </Button>
               ) : null}

               <Button onClick={form.cancel} extraProps={{ 'data-e2e': 'StageHugeForm:CancelButton' }}>
                  {form.showPage === StageHugeFormShowPage.Diff && !asIs
                     ? 'Edit'
                     : form.showPage === StageHugeFormShowPage.Update
                     ? 'Back'
                     : 'Cancel'}
               </Button>

               <Button
                  view={'action'}
                  onClick={form.update}
                  extraProps={{ 'data-e2e': 'StageHugeForm:UpdateButton' }}
                  disabled={form.showPage === StageHugeFormShowPage.Diff || form.isSubmitting}
               >
                  {form.isSubmitting ? (
                     <>
                        <FontAwesomeIcon icon={faSpinner} spin={true} /> Checking
                     </>
                  ) : (
                     'Review Changes'
                  )}
               </Button>
            </EditSharedNode>

            {form.content}
         </>
      ),
      [asIs],
   );

   if (!latestRawStage || !rawStage) {
      return null;
   }

   return (
      <StageHugeForm
         description={`Revert to changes from revision ${revisionId}`}
         latestRawStage={latestRawStage}
         onCancel={() => gotoHistoryEvent(revisionId)}
         rawStage={rawStage}
         mode={formMode}
      >
         {renderChildren}
      </StageHugeForm>
   );
};

StageRollbackInternalForm.displayName = 'StageRollbackInternalForm';
