import { useContext, useEffect, useMemo, useState } from 'react';

import { getHistoryEventStageRevision, HistoryEvent, HistorySpecRequestKeys } from '../../../../../models/ui';
import { useNetworkRequests } from '../../../../../redux';
import { RequestState } from '../../../../../redux/slices/network/model';
import { useArrayMemo } from '../../../../../utils';
import { HistoryListLoadedContext } from '../context';
import { useFetchHistoryEvent } from './useFetchHistoryEvent';
import { useHistoryList } from './useHistoryList';

export type RevisionsState = 'not_found' | 'error' | 'loading' | 'ok';

export type RevisionsCheckResult = {
   summary: {
      /**
       * самый приоритетный тип в ревизиях
       */
      type: RevisionsState;
      /**
       * ревизии для приоритетного типа
       */
      revisions: Set<number>;
   };
};

// TODO: по возможности легко добавить разбивку всех входящих ревизий по типам

/**
 * Определение суммарного статуса для списка ревизий истории стейджа
 */
export function useCheckRevisions(stageId: string, revisionRawIds: number[]) {
   const revisionIds = useArrayMemo(revisionRawIds);
   const { historyList } = useHistoryList(stageId);
   const [revisionsLoaded, setRevisionsLoaded] = useState<Set<number>>(() => new Set());

   const revisionsSet = useMemo(() => new Set(revisionIds), [revisionIds]);
   const eventsByRevisions = useMemo(() => {
      const record: Record<number, HistoryEvent> = {};
      for (const event of historyList) {
         const eventRevision = getHistoryEventStageRevision(event);
         if (revisionsSet.has(eventRevision)) {
            record[eventRevision] = event;
         }
      }
      return record;
   }, [historyList, revisionsSet]);

   const notFoundRevisions = useMemo(() => revisionIds.filter(id => !eventsByRevisions[id]), [
      eventsByRevisions,
      revisionIds,
   ]);

   const requestKeys = useMemo(
      () => revisionIds.map(id => HistorySpecRequestKeys.getOne({ id: stageId, revisionId: String(id) })),
      [revisionIds, stageId],
   );

   const requests = useNetworkRequests(requestKeys);

   const revisionsByRequests = useMemo(() => {
      const error: Set<number> = new Set();
      const loading: Set<number> = new Set();
      const ok: Set<number> = new Set();
      const unknown: Set<number> = new Set();
      revisionIds.forEach((id, i) => {
         const key = requestKeys[i];
         const state = requests[key]?.state;
         if (state === RequestState.ERROR) {
            error.add(id);
         } else if (state === RequestState.PENDING) {
            loading.add(id);
         } else if (state === RequestState.OK) {
            ok.add(id);
         } else {
            unknown.add(id);
         }
      });
      return { error, loading, ok, unknown };
   }, [requestKeys, requests, revisionIds]);

   const { fetchHistoryEvent } = useFetchHistoryEvent(stageId);

   useEffect(() => {
      // подгрузка новых событий
      let wasChange = false;
      const loaded = new Set(revisionsLoaded);
      for (const id of revisionIds) {
         const event = eventsByRevisions[id];
         if (!revisionsLoaded.has(id) && revisionsByRequests.unknown.has(id) && event) {
            // не загружали ранее, нет статуса запроса, событие найдено
            // только одна попытка
            // TODO: подумать над перезапросом
            fetchHistoryEvent(event);
            wasChange = true;
            loaded.add(id);
         }
      }

      if (wasChange) {
         setRevisionsLoaded(loaded);
      }
   }, [eventsByRevisions, fetchHistoryEvent, revisionIds, revisionsByRequests.unknown, revisionsLoaded]);

   const listLoaded = useContext(HistoryListLoadedContext);

   const summary: RevisionsCheckResult['summary'] = useMemo(() => {
      let result: RevisionsCheckResult['summary'];
      if (notFoundRevisions.length > 0 && listLoaded) {
         // ревизия не найдена, только если список уже загрузился полностью
         result = {
            type: 'not_found',
            revisions: new Set(notFoundRevisions),
         };
      } else if (revisionsByRequests.error.size > 0) {
         result = {
            type: 'error',
            revisions: revisionsByRequests.error,
         };
      } else if (revisionsByRequests.loading.size > 0) {
         result = {
            type: 'loading',
            revisions: revisionsByRequests.loading,
         };
      } else if (revisionsByRequests.unknown.size > 0) {
         result = {
            type: 'loading',
            revisions: revisionsByRequests.unknown,
         };
      } else if (revisionsByRequests.ok.size > 0) {
         result = {
            type: 'ok',
            revisions: revisionsByRequests.ok,
         };
      } else {
         // загрузка по умолчанию
         result = {
            type: 'loading',
            revisions: new Set(revisionIds),
         };
      }

      return result;
   }, [listLoaded, notFoundRevisions, revisionIds, revisionsByRequests]);

   return { summary };
}
