import { useCallback, useEffect, useRef, useState } from 'react';

import { fromQuery } from '@yandex-infracloud-ui/libs';
import { useDispatch, useSelector } from 'react-redux';
import { useLocation } from 'react-router';

import { HistoryRequestKeys } from '../../../../../models/ui/history';
import { EObjectType } from '../../../../../proto-typings';
import { fetchHistory, selectHistoryContinuationTokens, useRequestControl, useStage } from '../../../../../redux';

export function useFetchHistory(stageId: string) {
   const [isLoaded, setIsLoaded] = useState(true);
   const { rawStage } = useStage(stageId);

   const { id = '', uuid = '' } = rawStage?.meta ?? {};
   const requestKey = HistoryRequestKeys.getOne({ id: stageId });

   const historyTokens = useSelector(selectHistoryContinuationTokens);
   const lastToken: string | null | undefined = historyTokens[uuid];
   const lastTokenRef = useRef<string | null | undefined>(undefined);

   useEffect(() => {
      lastTokenRef.current = lastToken;
   }, [lastToken]);

   const dispatch = useDispatch();

   const fetchHistoryRecords = useCallback(
      ({ init = false, limit }: { init?: boolean; limit?: number } = {}) => {
         if (!uuid || lastTokenRef.current === null) {
            return;
         }

         dispatch(
            fetchHistory.withRequestKey(requestKey).withMeta({ reset: Boolean(init) })({
               type: EObjectType.OT_STAGE,
               id,
               uuid,
               limit,
               continuationToken: init ? undefined : lastTokenRef.current,
               // важно — запрашиваем только 2 нужных поля
               paths: ['/spec/revision', '/spec/revision_info/description'],
            }),
         );
      },
      [dispatch, id, requestKey, uuid],
   );

   const availableWork = useRef<(() => void) | null>(null);
   const { activate } = useRequestControl([requestKey], {
      onSuccess: () => {
         if (availableWork.current) {
            availableWork.current();
         }
      },
      onError: () => {
         setIsLoaded(true);
      },
   });

   const location = useLocation();
   const search = fromQuery(location.search);
   // кастомный лимит для очень больших стейджей
   const limit = search.limit ? Number(search.limit) : 10;

   const fetchHistoryParts = useCallback(
      ({ init = false, size = 100 }: { init?: boolean; size?: number } = {}) => {
         let loadedSize = 0;
         const rFetchHistory = ({ init: rInit = false }: { init?: boolean } = {}) => {
            // задержка нужна, чтобы обновление редакса дошло до этого хука и lastTokenRef.current обновился в эффекте выше
            availableWork.current = () =>
               window.setTimeout(() => {
                  loadedSize += limit;

                  if (lastTokenRef.current !== null && loadedSize < size) {
                     rFetchHistory();
                  } else {
                     setIsLoaded(true);
                  }
               }, 300);

            activate(() => {
               fetchHistoryRecords({ init: rInit, limit });
            }, [requestKey]);
         };

         setIsLoaded(false);

         rFetchHistory({ init });
      },
      [activate, fetchHistoryRecords, limit, requestKey],
   );

   return {
      fetchHistoryParts,
      isLoaded,
   };
}
