import { useEffect, useState } from 'react';

import { getSetDifference } from '@yandex-infracloud-ui/libs';
import { useDispatch } from 'react-redux';

import { fetchTicketsData } from '../../../../../redux/slices/startrek/startrek';
import { parseTextToObjects, TextParserResult, TextParserType } from '../../../../../utils/parsers/parseTextToObjects';
import { useHistoryList } from './useHistoryList';

export function useHistoryEventDescriptions(stageId: string) {
   const [historyDescriptions, setHistoryDescriptions] = useState<Map<number, TextParserResult[]>>(() => new Map());
   const [ticketIds, setTicketIds] = useState<Set<string>>(() => new Set());
   const [loadedStartrekIds, setLoadedStartrekIds] = useState<Set<string>>(() => new Set());

   const { historyList } = useHistoryList(stageId);

   const dispatch = useDispatch();

   useEffect(() => {
      let wasChanged = false;
      const historyDescriptionsMap = new Map(historyDescriptions);
      const startrekTickets = new Set(ticketIds);

      for (const historyElement of historyList) {
         const revisionId = historyElement?.result?.spec?.revision ?? null;
         const description = historyElement?.result?.spec?.revision_info?.description ?? null;

         if (!historyDescriptionsMap.has(revisionId) && description) {
            wasChanged = true;

            const parsedString = parseTextToObjects(description, {
               startrekTicketUrl: true,
               links: true,
               startrekTicket: true,
            });

            historyDescriptionsMap.set(revisionId, parsedString);

            for (const { type, text } of parsedString) {
               if (type === TextParserType.STARTREK_TICKET) {
                  startrekTickets.add(text);
               }
            }
         }
      }

      if (wasChanged) {
         // при изменении обновляем описания
         setHistoryDescriptions(historyDescriptionsMap);
         setTicketIds(startrekTickets);

         const { added } = getSetDifference(loadedStartrekIds, startrekTickets);

         if (added.size > 0) {
            // подгрузка новых тикетов

            dispatch(
               fetchTicketsData({
                  ids: [...added],
               }),
            );
            setLoadedStartrekIds(new Set(startrekTickets));
         }
      }
   }, [dispatch, historyDescriptions, historyList, loadedStartrekIds, ticketIds]);

   return { historyDescriptions };
}
