import { useCallback, useEffect, useMemo, useState } from 'react';

import { TIMES_IN_MS, UpdateTimerMode, useUpdateTimer } from '@yandex-infracloud-ui/libs';
import { useDispatch } from 'react-redux';

import { isRetryingXRayStatus } from '../../../models/ui/stage/XRay';
import { useStage } from '../../../redux';
import { useXRayStatus } from '../../../redux/hooks/useXRayStatus';
import { fetchXRayStatus } from '../../../redux/slices/xray/xray';

const MAX_XRAY_RETRYING_COUNT = 5;
const increment = (n: number) => n + 1;
export function useUpdateXRayStatus(stageId: string) {
   const { stage, rawStage } = useStage(stageId);
   const { revision, id } = stage || {};
   const { uuid } = rawStage?.meta ?? {};
   const { xRayStatus } = useXRayStatus(stageId);

   const dispatch = useDispatch();
   const [retryingCount, setRetryingCount] = useState(0);
   const availableRetry = retryingCount < MAX_XRAY_RETRYING_COUNT;
   const statusThunk = useMemo(() => (id && uuid && revision ? fetchXRayStatus({ id, uuid, revision }) : null), [
      id,
      revision,
      uuid,
   ]);

   useEffect(() => {
      setRetryingCount(0);
   }, [statusThunk]);

   const updateXRayStatus = useCallback(() => {
      if (statusThunk) {
         if (availableRetry && isRetryingXRayStatus(xRayStatus.analysisStatus)) {
            dispatch(statusThunk);
            setRetryingCount(increment);
         }
      }
   }, [statusThunk, availableRetry, xRayStatus.analysisStatus, dispatch]);

   useEffect(() => updateXRayStatus(), [updateXRayStatus]);

   useUpdateTimer({
      callback: updateXRayStatus,
      fast: TIMES_IN_MS.Minute,
      mode: UpdateTimerMode.Slow,
      slow: TIMES_IN_MS.Minute * 5,
   });
}
