import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';

import {
   ButtonLink,
   isEmpty,
   SwitchRoutes,
   TIMES_IN_MS,
   UpdateTimerMode,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs';
import { useDispatch } from 'react-redux';
import { Route, RouteComponentProps } from 'react-router';

import { SharedNodeProvider } from '../../../components/lib';
import { YpErrorTooltip } from '../../../components/network';
import { formPlates } from '../../../components/stage-huge-form/components/plates';
import { urlBuilder } from '../../../models';
import LoadingPage from '../../../old-code/components/LoadingPage/LoadingPage';
import { getStages, useNetworkErrors, useStage, useStagePermissions } from '../../../redux';
import { ROUTES } from './_routes';
import { RemoveStageButton, StageBreadcrumbs, StageNotFound, StageTabs } from './components';
import { StageStatusSummaryLine } from './components/StageStatusSummaryLine/StageStatusSummaryLine';
import { StageRequestKeys } from '../../../models/ui';
import {
   getEmptyStatusViewStore,
   StatusViewContext,
   StatusViewStoreUpdater,
   StatusViewUpdateContext,
   UpdateStatusViewStore,
} from './status/context';
import { getStageStatusView, StatusTtl } from '../../../models/ui/status';
import { createKey } from '../../../utils';
import { useUpdateXRayStatus } from './hooks';
import { useUpdateStageProblems } from './status/hooks';

import classes from './index.module.css';

const STAGE_UPDATE_PERIOD = TIMES_IN_MS.Second * 10;

function useUpdateStage(stageId: string) {
   const stageRequestKey = StageRequestKeys.getOne({ id: stageId });
   const dispatch = useDispatch();
   const loadStage = useCallback(() => {
      dispatch(
         getStages.withRequestKey(stageRequestKey)({
            objectIds: [stageId],
            paths: s => [s.meta, s.labels, s.spec, s.status],
         }),
      );
   }, [dispatch, stageRequestKey, stageId]);

   useUpdateTimer({
      callback: loadStage,
      fast: STAGE_UPDATE_PERIOD,
      slow: TIMES_IN_MS.Minute,
      mode: UpdateTimerMode.Fast,
   });
}

/**
 * компонент не должен перерендериваться после первоначального монтирования
 */
export const StageLayoutContent: React.FC<{
   sharedNodeRef: React.RefObject<HTMLDivElement>;
   canEdit: boolean;
   routeProps: RouteComponentProps<{ stageId: string }>;
   updateStatusViewStore: StatusViewStoreUpdater;
   stageId: string;
   projectId: string;
}> = React.memo(({ sharedNodeRef, canEdit, routeProps, updateStatusViewStore, stageId, projectId }) => {
   const { match, history } = routeProps;
   const userHasAccessToEdit = useStagePermissions(stageId);

   const handleRemovedStage = useCallback(() => {
      history.push(urlBuilder.project(projectId));
   }, [history, projectId]);

   return (
      <div className={classes.content} data-e2e={'StageIndexPage:content'}>
         <SharedNodeProvider
            common={
               canEdit && (
                  <>
                     <Route path={[`${match.path}/edit`, `${match.path}/config`]}>
                        {routeChildrenProps =>
                           routeChildrenProps.match || !userHasAccessToEdit.edit ? null : (
                              <ButtonLink to={urlBuilder.stageEdit(stageId)} data-e2e={'EditStageButton'}>
                                 Edit
                              </ButtonLink>
                           )
                        }
                     </Route>
                     <RemoveStageButton onSuccess={handleRemovedStage} stageId={stageId} projectId={projectId} />
                  </>
               )
            }
            nodeRef={sharedNodeRef}
         >
            <StatusViewUpdateContext.Provider value={updateStatusViewStore}>
               <SwitchRoutes>{ROUTES}</SwitchRoutes>
            </StatusViewUpdateContext.Provider>
         </SharedNodeProvider>
      </div>
   );
});

export const StageIndexPage: React.FC<RouteComponentProps<{ stageId: string }>> = React.memo(routeProps => {
   const { match } = routeProps;
   const { stageId } = match.params;
   const { stage, meta, stageStatus, stageStatusLabel } = useStage(stageId);

   useUpdateStage(stageId);
   useUpdateXRayStatus(stageId);
   useUpdateStageProblems(stageId);

   const sharedNodeRef = useRef<HTMLDivElement>(null);

   const stageRequestKey = StageRequestKeys.getOne({ id: stageId });
   const networkErrors = useNetworkErrors([stageRequestKey]);
   const stageError = networkErrors[stageRequestKey];

   const canEdit = true; // stage && !stageError; // TODO

   const needDiskIsolation = useMemo(() => {
      const duWithoutDiskIsolation = stage?.deployUnits?.filter(du => !du.tempDiskIsolation);

      return !isEmpty(duWithoutDiskIsolation);
   }, [stage]);

   const [statusViewStore, setStatusViewStore] = useState(getEmptyStatusViewStore());

   const updateStatusViewStore: StatusViewStoreUpdater = useCallback((updateStore: UpdateStatusViewStore) => {
      setStatusViewStore(updateStore);
   }, []);

   const stageStoreKey = createKey({ stageId });

   const { lastUpdateTimestamp = 0, mergeStageStatusLabel } = statusViewStore.stages[stageStoreKey] ?? {};
   const { calculate: calculatedStageStatus } = mergeStageStatusLabel ?? {};

   useEffect(() => {
      if (stage && stageStatus && stageStatusLabel) {
         const now = Date.now();
         const ttl = calculatedStageStatus ? StatusTtl.stage : STAGE_UPDATE_PERIOD;
         const isViewOld = now - lastUpdateTimestamp >= ttl;
         if (isViewOld) {
            const { stageStatusView } = getStageStatusView({
               stage,
               stageStatus,
               stageStatusLabel,
               deployUnitData: [],
            });

            updateStatusViewStore(store => ({ ...store, stages: { [stageStoreKey]: stageStatusView } }));
         }
      }
   }, [
      calculatedStageStatus,
      lastUpdateTimestamp,
      stage,
      stageStatus,
      stageStatusLabel,
      stageStoreKey,
      updateStatusViewStore,
   ]);

   if (meta.isProcessing) {
      return <LoadingPage />;
   }

   if (!stage && meta.isNotFound) {
      return <StageNotFound stageId={stageId} />;
   }

   if (!stage) {
      return (
         <p>
            Fetching stage: error
            {stageError && <YpErrorTooltip error={stageError.error!} request={stageError.request} />}
         </p>
      );
   }

   const pageHeader = (
      <>
         <div className={classes.headerPanel}>
            <div className={classes.breadcrumbs} data-e2e={'StageIndexPage:Breadcrumbs'}>
               <StageBreadcrumbs stageId={stageId} projectId={stage?.project?.id} />
            </div>
            <div>
               <StatusViewContext.Provider value={statusViewStore}>
                  <StageStatusSummaryLine stageId={stageId} />
               </StatusViewContext.Provider>
            </div>
            <div className={classes.buttons} ref={sharedNodeRef} />
         </div>

         <div className={classes.tabs} data-e2e={'StageIndexPage:Tabs'}>
            <StageTabs stageId={stageId} />
         </div>

         {needDiskIsolation ? (
            <div className={classes.notifications} data-e2e={'StageIndexPage:Notifications'}>
               <div className={classes.warning}>{formPlates['DEPLOY-3611']}</div>
            </div>
         ) : null}
      </>
   );

   return (
      <div className={classes.page}>
         {/* Страницы создания/просмотра/редактирования релизного правила не должны содержать общую шапку */}
         <Route path={`${match.path}/release_rules/:rest`}>
            {routeChildrenProps => (routeChildrenProps.match ? null : pageHeader)}
         </Route>

         <StageLayoutContent
            stageId={stageId}
            projectId={stage.project?.id ?? ''}
            sharedNodeRef={sharedNodeRef}
            canEdit={canEdit}
            routeProps={routeProps}
            updateStatusViewStore={updateStatusViewStore}
         />
      </div>
   );
});

StageIndexPage.displayName = 'StageIndexPage';
