import { useTitle } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useState } from 'react';
import { RouteComponentProps } from 'react-router';
import { Link } from 'react-router-dom';
import { EXTERNAL_LINKS, urlBuilder } from '../../../../../models';
import Select from '../../../../../old-code/components/Select/Select';
import { useStage } from '../../../../../redux';
import { useConfig } from '../../../../../services';

import classes from './index.module.css';

interface ConfigItem {
   deployUnit: string;
   title: string;
   itype: string;
   stageId: string;
}

export const StageMonitoringLogsPage: React.FC<RouteComponentProps<{ stageId: string }>> = React.memo(
   ({ location, match }) => {
      const { stageId } = match.params;
      const { rawStage } = useStage(stageId, true);

      useTitle(`Monitoring Logs / ${stageId}`);

      const config = useConfig();

      const [monitoringLocations, setMonitoringLocations] = useState<string[]>([]);
      const [monitoringActiveWorkload, setMonitoringActiveWorkload] = useState<ConfigItem | null>(null);

      // TODO дальше идет портированный на TS, но не переписанный нормально код
      const deployUnits = rawStage?.status?.deploy_units ?? {};

      const deployUnitNames: string[] = [];
      // eslint-disable-next-line react-hooks/exhaustive-deps
      const configItems: Record<string, ConfigItem> = {};
      const availableLocations: Record<string, Record<string, boolean>> = {};

      Object.keys(deployUnits)
         .sort()
         .forEach(deployUnitName => {
            const deployUnit = deployUnits[deployUnitName];
            const set = deployUnit.replica_set ?? deployUnit.multi_cluster_replica_set;
            const clusterStatuses = set?.cluster_statuses ?? {};
            for (const cluster of Object.keys(clusterStatuses)) {
               if (!(deployUnitName in availableLocations)) {
                  availableLocations[deployUnitName] = {};
               }
               availableLocations[deployUnitName][cluster] = true;
            }
            deployUnitNames.push(deployUnitName);
            configItems[deployUnitName] = {
               deployUnit: deployUnitName,
               title: deployUnitName,
               itype: deployUnit.yasm_itype,
               stageId,
            };
         });

      const availableGeoLocations: Record<string, Record<string, any[]>> = {};
      for (const deployUnitName of Object.keys(availableLocations)) {
         const deployUnitLocations = availableLocations[deployUnitName];
         if (!availableGeoLocations.hasOwnProperty(deployUnitName)) {
            availableGeoLocations[deployUnitName] = {};
         }
         for (const duLocation of Object.keys(deployUnitLocations)) {
            const overrideGeoLocation = config?.overrideGeoLocation ?? {};
            const yasmGeoLocation = overrideGeoLocation[duLocation] ?? duLocation;
            const value = availableGeoLocations[deployUnitName][yasmGeoLocation];
            availableGeoLocations[deployUnitName][yasmGeoLocation] = [...(value || []), duLocation];
         }
      }

      const handleLocationsChange = useCallback((value: string[]) => {
         setMonitoringLocations(value.sort());
      }, []);

      const handleDeployUnitChange = useCallback(
         (value: string) => {
            setMonitoringActiveWorkload(configItems[value]);
         },
         [configItems],
      );

      // region render
      if (!rawStage) {
         return <div data-test={'view-stage--monitoring'} />;
      }
      if (deployUnitNames.length === 0) {
         return (
            <div data-test={'view-stage--monitoring'}>
               <div className={classes.noDeployUnits} data-test={'no-deploy-units'}>
                  Please define at least one Deploy Unit
               </div>
            </div>
         );
      }

      const activeWorkload =
         monitoringActiveWorkload && configItems[monitoringActiveWorkload.title]
            ? monitoringActiveWorkload
            : configItems[deployUnitNames[0]];

      const currentDeployUnitName = activeWorkload.deployUnit;

      const configItemToSelectItem = (configItem: ConfigItem) => ({
         key: `key-monitoring-${configItem.deployUnit}`,
         val: configItem.deployUnit,
         text: configItem.title,
      });

      const controlItems = deployUnitNames.map(duName => {
         const monitoringConfig = configItems[duName];

         return configItemToSelectItem(monitoringConfig);
      });

      // if 'constructor' object keys only #DEPLOY-5610
      const getDeployUnitLocations = (deployUnitName: string) =>
         availableGeoLocations.hasOwnProperty(deployUnitName) ? availableGeoLocations[deployUnitName] : {};

      const getGeoText = (deployUnitName: string) => (geoLocation: string) => {
         const deployUnitLocations = getDeployUnitLocations(deployUnitName);
         const locations = deployUnitLocations?.[geoLocation] ?? [];

         return `${locations.join('/').toUpperCase()}${locations.length > 1 ? `(${geoLocation})` : ''}`;
      };

      const locationItems = Object.keys(getDeployUnitLocations(currentDeployUnitName))
         .sort()
         .map(e => ({ key: e, val: e, text: getGeoText(currentDeployUnitName)(e) }));

      const filterLocations = (deployUnitName: string) =>
         monitoringLocations.length > 0 ? monitoringLocations : Object.keys(getDeployUnitLocations(deployUnitName));

      const panelParams: Record<string, string> = {
         'p.Stage': activeWorkload.stageId,
         'p.DeployUnit': activeWorkload.deployUnit,
      };

      filterLocations(currentDeployUnitName).forEach((loc, i) => {
         panelParams[`p.Location[${i}]`] = loc[0].toUpperCase() + loc.substring(1);
      });

      return (
         <div data-test={'view-stage--monitoring'}>
            <div className={classes.filters}>
               <div data-test={'monitoring-select'}>
                  <Select
                     val={currentDeployUnitName}
                     items={controlItems}
                     onChange={handleDeployUnitChange}
                     button={{
                        text: `DU: ${currentDeployUnitName}`,
                     }}
                  />
               </div>

               <div>
                  <Select
                     type={'check'}
                     val={monitoringLocations.sort()}
                     button={{
                        text: `Location: ${
                           monitoringLocations.length
                              ? monitoringLocations.map(e => getGeoText(currentDeployUnitName)(e)).join(', ')
                              : 'all available'
                        }`,
                     }}
                     items={locationItems}
                     onChange={handleLocationsChange}
                  />
               </div>

               <div>
                  Unified Agent Monitoring is enabled on Runtime version 7 or higher.&nbsp;
                  <Link to={urlBuilder.stageEdit(activeWorkload.stageId, { duId: activeWorkload.deployUnit })}>
                     Check your Deploy Unit Settings
                  </Link>
               </div>
            </div>
            <iframe
               src={EXTERNAL_LINKS.logsPanel(panelParams)}
               title={'Monitoring frame'}
               className={classes.iframe}
               data-test={'monitoring-iframe'}
            />
         </div>
      );
      // endregion
   },
);

StageMonitoringLogsPage.displayName = 'StageMonitoringLogsPage';
