import React, { useCallback, useContext, useEffect, useRef, useState } from 'react';

import { DISMISS_REASON, modalService, toasts, toQuery, useDismounted, useTitle } from '@yandex-infracloud-ui/libs';
import { FormikHelpers, FormikProps } from 'formik';
import { Spin } from 'lego-on-react';
import { useDispatch, useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { finalize, takeUntil } from 'rxjs/operators';

import { FormPage } from '../../../../components';
import { handleApiError, NotificationsContext, urlBuilder } from '../../../../models';
import {
   DefaultReleaseRuleType,
   ReleaseRule,
   ReleaseRuleFormParams as FormParams,
   ReleaseRuleType,
} from '../../../../models/ui';
import { fetchReleaseRule, RootState, selectRule, useStage } from '../../../../redux';
import { ypApi } from '../../../../services';
import { RuleRemoveModal } from '../deploy-tickets/components/RuleRemoveModal/RuleRemoveModal';
import { validationSchema } from './components/ReleaseRuleForm/models';
import { ReleaseRuleForm } from './components/ReleaseRuleForm/ReleaseRuleForm';
import { ReleaseRuleReadonlyPage } from './components/ReleaseRuleReadonlyPage/ReleaseRuleReadonlyPage';
import { useReleaseRuleFormContext } from './models';
import { DevJson } from '../../../../components/lib';

import classes from './_ruleId.module.scss';

type PageType = 'view' | 'edit';

export const ReleaseRulePage: React.FC<RouteComponentProps<{ stageId: string; ruleId: string }>> = React.memo(
   ({ match, history }) => {
      const { stageId, ruleId } = match.params;

      // region hooks
      const notifications = useContext(NotificationsContext);

      const [page, setPage] = useState('view' as PageType);

      const rule = useReleaseRule(ruleId);
      const { stage } = useStage(stageId);
      // Проброс данных из stage внутрь кастомных полей формы
      const formContext = useReleaseRuleFormContext(stageId, false);
      const formRef = useRef<FormikProps<FormParams>>();
      const dismounted = useDismounted();
      // endregion

      // region effects
      const title = page === 'edit' ? 'Edit release rule' : `Rule: ${ruleId} / ${stageId}`;

      useTitle(title);
      // endregion

      // region handlers
      const handleFieldChange = useCallback(
         (field: string | number | symbol, v: any) => {
            if (field === 'type') {
               const t = v as ReleaseRuleType;
               const search = { type: t === DefaultReleaseRuleType ? undefined : t };

               history.push({ search: toQuery(search) });
            }
         },
         [history],
      );

      const goToReleaseList = useCallback(() => history.push(urlBuilder.stageDeployTickets(stageId)), [
         history,
         stageId,
      ]);

      const dispatch = useDispatch();

      const handleSubmit = useCallback(
         (v: FormParams, helpers: FormikHelpers<FormParams>) => {
            ypApi
               .updateReleaseRule(v)
               .pipe(
                  finalize(() => helpers.setSubmitting(false)),
                  takeUntil(dismounted),
               )
               .subscribe(() => {
                  toasts.success(`Release rule ${v.id} updated`);
                  dispatch(fetchReleaseRule(ruleId));
                  setPage('view');
               }, handleApiError('Updating release rule', notifications));
         },
         [dismounted, setPage, notifications, dispatch, ruleId],
      );

      const handleRemoveRule = useCallback(() => {
         modalService
            .open(RuleRemoveModal, { rule: rule as ReleaseRule })
            .pipe(takeUntil(dismounted))
            .subscribe(goToReleaseList, reason => {
               if (reason !== DISMISS_REASON) {
                  toasts.apiError('Removing rule', reason);
               }
            });
      }, [dismounted, goToReleaseList, rule]);
      // endregion

      // region render

      if (rule === null) {
         return (
            <div className={classes.spinner}>
               <Spin progress size={'l'} />
            </div>
         );
      }

      if (stage && rule && page === 'view') {
         return (
            <ReleaseRuleReadonlyPage
               onEdit={() => setPage('edit')}
               onRemove={handleRemoveRule}
               stageId={stageId}
               projectId={stage.project!.id}
               rule={rule}
            />
         );
      }

      // noinspection PointlessBooleanExpressionJS
      return (
         <>
            <FormPage
               cancelButtonText={'Cancel'}
               subTitle={'Edit release rule'}
               submitButtonText={'Save'}
               title={`${title} for stage "${stageId}"`}
               formRef={formRef}
               loading={rule === null ? `Rule "${ruleId}" is loading` : null}
               initialValues={rule?.form}
               validationSchema={validationSchema}
               formContext={formContext}
               onFieldChange={handleFieldChange}
               onSubmit={handleSubmit}
               onCancel={() => setPage('view')}
            >
               {() => <ReleaseRuleForm />}
            </FormPage>

            <DevJson summary={'Rule'}>{rule}</DevJson>
         </>
      );
      // endregion
   },
);

ReleaseRulePage.displayName = 'ReleaseRulePage';

function useReleaseRule(ruleId: string): ReleaseRule | null {
   const dispatch = useDispatch();

   const ruleSelector = useCallback((s: RootState) => selectRule(s, ruleId), [ruleId]);
   const rule = useSelector(ruleSelector);

   useEffect(() => {
      if (!rule) {
         dispatch(fetchReleaseRule(ruleId));
      }
   }, [dispatch, rule, ruleId]);

   return rule ?? null;
}
