import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import {
   ExtendedFieldConfig,
   FieldLayout2,
   FormButton,
   InputField2,
   isEmpty,
   RemoveButton,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback } from 'react';
import { StagePart } from '../../../../../../components';
import { defaultDockerPatch, DockerFormParams, DockerPatch } from '../../../../../../models/ui';

import { getSequentialId } from '../../../../../../utils';
import { TargetSelect } from './components/TargetSelect';

import classes from './DockerParamsField.module.css';

interface PatchesTableRow {
   target: ReactNode;
   patchName: ReactNode;
   buttons: ReactNode;
}

const patchesColumns: TableColumnConfig<PatchesTableRow>[] = [
   { id: 'target', name: 'Target' },
   { id: 'patchName', name: 'Patch name' },
   { id: 'buttons', name: '' },
];

export const DockerParamsField = React.memo((props: ExtendedFieldConfig<any, DockerFormParams | null>) => {
   const { field, onChange, disabled } = useExtendedField(props);
   const { value } = field;
   const { required } = props;

   const handlePatchesChange = useCallback(
      (patches: DockerPatch[]) => {
         if (value) {
            onChange({
               ...value,
               patches,
            });
         }
      },
      [onChange, value],
   );

   const handleAddPatch = useCallback(() => {
      handlePatchesChange([...(value?.patches ?? []), { ...defaultDockerPatch }]);
   }, [handlePatchesChange, value]);

   const handleDeletePatch = useCallback(
      (i: number) => {
         const patches = [...(value?.patches ?? [])];

         patches.splice(i, 1);

         handlePatchesChange(patches);
      },
      [handlePatchesChange, value],
   );

   const handlePatchChange = useCallback(
      (i: number, newPatch: DockerPatch) => {
         handlePatchesChange((value?.patches ?? []).map((p, index) => (i === index ? newPatch : p)));
      },
      [handlePatchesChange, value],
   );

   const handlePatchTargetSelect = useCallback(
      (i: number, v: StagePart | null) => {
         const patch = value?.patches[i];

         if (patch) {
            const siblingIds = (value?.patches ?? []).filter(p => p.id !== patch.id).map(p => p.id);
            const defaultId = v?.resource?.id
               ? getSequentialId(`patch-${v.resource.id}`, siblingIds)
               : getSequentialId(`patch`, siblingIds);

            handlePatchChange(i, {
               ...patch,
               deployUnitId: v?.deployUnit ?? '',
               boxId: v?.box ?? '',
               id: patch.id && !isEmpty(patch.id) ? patch.id : defaultId,
            });
         }
      },
      [handlePatchChange, value],
   );

   return (
      <FieldLayout2 {...props} required={false}>
         <InputField2
            name={`${field.name}.image`}
            label={'Docker image'}
            disabled={disabled}
            required={required}
            controlProps={{ placeholder: 'Enter docker image' }}
         />

         <FieldLayout2
            name={`${field.name}.patches`}
            label={'Deploy patches'}
            disabled={disabled}
            required={required}
            // выводим общую ошибку массива только, когда массив пустой
            // (например, "You should add at least one patch")
            // если массив не пустой, то выводим ошибки в отдельных полях
            hideErrors={value?.patches && !isEmpty(value.patches)}
         >
            <Table
               columns={patchesColumns}
               data={(value?.patches ?? []).map((patch, i) => ({
                  target: (
                     <div className={classes.target} data-e2e={`DeployPatches:${i + 1}:ResourceTarget`}>
                        <FieldLayout2 name={`${field.name}.patches[${i}].boxId`} label={null}>
                           <TargetSelect
                              id={patch.boxId}
                              deployUnit={patch.deployUnitId}
                              onSelect={(v: StagePart | null) => handlePatchTargetSelect(i, v)}
                              disabled={disabled}
                           />
                        </FieldLayout2>
                     </div>
                  ),
                  patchName: (
                     <div className={classes.patchName} data-e2e={`DeployPatches:${i + 1}:Name`}>
                        <InputField2
                           name={`${field.name}.patches[${i}].id`}
                           label={null}
                           disabled={disabled}
                           controlProps={{ placeholder: 'Enter patch name' }}
                        />
                     </div>
                  ),
                  buttons: (
                     <div data-e2e={`DeployPatches:${i + 1}:RemovePatch`}>
                        <RemoveButton onClick={() => handleDeletePatch(i)} />
                     </div>
                  ),
               }))}
               emptyMessage={'No patches. You should add at least one.'}
               verticalAlign={'top'}
            />

            {/* TODO: AddButton #DEPLOY-4915 */}
            <div data-e2e={'DeployPatches:AddPatch'}>
               <FormButton onClick={handleAddPatch} skipLeftSpace={true} icon={faPlus}>
                  Add patch
               </FormButton>
            </div>
         </FieldLayout2>
      </FieldLayout2>
   );
});

DockerParamsField.displayName = 'DockerParamsField';
