import { EnumField2, ExtendedFieldConfig, InputField2, TextField2 } from '@yandex-infracloud-ui/libs';
import { array, boolean, mixed, object, Schema, string } from 'yup';

import {
   DockerFormParams,
   DockerPatch,
   ReleaseRuleFormParams,
   ReleaseRuleType,
   ResourceType,
   SandboxFormParams,
   SandboxPatch,
} from '../../../../../../models/ui';

import { ReleaseRuleFormContext } from '../../models';

import { AutocommitField } from '../AutocommitField/AutocommitField';
import { DockerParamsField } from '../DockerParamsField/DockerParamsField';
import { SandboxParamsField } from '../SandboxParamsField/SandboxParamsField';
import { ReleaseTypesField } from './components/ReleaseTypesField/ReleaseTypesField';

/* eslint-disable no-template-curly-in-string */
const trimmedStringSchema = (label: string) =>
   string()
      .label(label)

      .test('trimmed', '${path} should not include whitespaces', function Trimmed(v) {
         if (v && v !== v.trim()) {
            return this.createError({
               message: `${'${path} should not include whitespaces'} (${v})`,
            });
         }

         return true;
      });

const hasOnlyUniqueValues = (items: any[]) => new Set(items).size === items.length;

export const sandboxParamsValidationSchema = object<SandboxFormParams>({
   attributes: array().of(
      object({
         key: trimmedStringSchema('Attribute key')
            .min(1)
            .max(128)
            .matches(/^[\w-.]+$/),
         value: string().label('Attribute value').min(1).max(128),
      }),
   ),
   taskType: trimmedStringSchema('Sandbox task type').required(),
   resources: array().of(string()).required(),
   patches: array()
      .of(
         object<SandboxPatch>({
            deployGroupMark: string().notRequired(),
            deployUnitId: string().when('type', {
               is: ResourceType.DynamicResource,
               then: string().notRequired(),
               otherwise: string().label('Deploy Unit ID').required('Target is required (Deploy Unit ID, Resource ID)'),
            }),
            id: trimmedStringSchema('Patch name').required(),
            ref: string().label('Target resource').required('Target is required (Deploy Unit ID, Resource ID)'),
            type: string()
               .label('Resource type')
               .oneOf([
                  ResourceType.Layer,
                  ResourceType.StaticResource,
                  ResourceType.DynamicResource,
               ]) as Schema<ResourceType>,
            resourceType: trimmedStringSchema('Resource type').required(),
         }),
      )
      .min(1, 'You should add at least one patch')
      .test('uniqueId', 'Patches should have unique IDs', (v: SandboxPatch[]) => hasOnlyUniqueValues(v.map(p => p.id))),
});

export const dockerParamsValidationSchema = object<DockerFormParams>({
   image: trimmedStringSchema('Docker image').required(),
   patches: array()
      .label('Patches')
      .of(
         object<DockerPatch>({
            id: trimmedStringSchema('Patch name').required(),
            deployUnitId: string().label('Deploy Unit ID').required('Target is required (Deploy Unit ID, Box ID)'),
            boxId: string().label('Box ID').required('Target is required (Deploy Unit ID, Box ID)'),
         }),
      )
      .min(1, 'You should add at least one patch')
      .test('uniqueId', 'Patches should have unique IDs', (v: DockerPatch[]) => hasOnlyUniqueValues(v.map(p => p.id))),
});

export const validationSchema = object<ReleaseRuleFormParams>({
   autocommit: boolean().label('Autocommit ticket').required(),

   id: trimmedStringSchema('Release rule ID').required(),

   description: string().label('Description'),

   releaseTypes: array().of(
      trimmedStringSchema('Release type')
         .min(1)
         .matches(/^[a-z0-9-]+$/),
   ),

   type: string()
      .label('Release rule type')
      .oneOf([ReleaseRuleType.Sandbox, ReleaseRuleType.Docker])
      .typeError('${type} is required field')
      .required() as Schema<ReleaseRuleType>,

   docker: mixed().when('type', {
      is: ReleaseRuleType.Docker,
      then: dockerParamsValidationSchema.required(),
      otherwise: mixed(),
   }),

   sandbox: mixed().when('type', {
      is: ReleaseRuleType.Sandbox,
      then: sandboxParamsValidationSchema.required(),
      otherwise: mixed(),
   }),
});
/* eslint-enable no-template-curly-in-string */

export const fields: Array<ExtendedFieldConfig<ReleaseRuleFormParams>> = [
   {
      as: InputField2,
      name: 'id',
      label: 'Release rule ID',
      placeholder: 'Enter release rule ID',
      required: true,
      readonly: (_, form, context: ReleaseRuleFormContext) => !context.isNew,
      readonlyDots: true,
   },
   {
      as: TextField2,
      name: 'description',
      label: 'Description',
      placeholder: 'Enter release rule description in free form',
   },
   {
      as: ReleaseTypesField,
      name: 'releaseTypes',
      label: 'Release types',
      placeholder: 'Any',
      required: true,
   },
   {
      as: AutocommitField,
      name: 'autocommit',
      label: 'Autocommit tickets',
   },
   {
      as: EnumField2,
      name: 'type',
      label: 'Release rule type',
      required: true,
      readonly: true,
      readonlyDots: true,
      controlProps: {
         options: [
            { value: ReleaseRuleType.Sandbox, title: 'Sandbox' },
            { value: ReleaseRuleType.Docker, title: 'Docker' },
         ],
      },
   },

   {
      as: SandboxParamsField,
      name: 'sandbox',
      label: 'Sandbox settings',
      bigLabel: true,
      hideErrors: true,
      required: (_, form) => form.values.type === ReleaseRuleType.Sandbox,
      hidden: (_, form) => form.values.type !== ReleaseRuleType.Sandbox,
   },

   {
      as: DockerParamsField,
      name: 'docker',
      label: 'Docker settings',
      bigLabel: true,
      hideErrors: true,
      required: (_, form) => form.values.type === ReleaseRuleType.Docker,
      hidden: (_, form) => form.values.type !== ReleaseRuleType.Docker,
   },
];
