import React, { useMemo } from 'react';

import { faPencil, faTrashAlt } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import { EMPTY_VALUE, isEmpty } from '@yandex-infracloud-ui/libs';

import { DevJson } from '../../../../../../components/lib';
import { Page } from '../../../../../../components/layout/Page/Page';
import { urlBuilder } from '../../../../../../models';
import { ReleaseRule, ReleaseRuleType } from '../../../../../../models/ui';
import { Breadcrumbs } from '../../../../../../old-code/components/Breadcrumbs/Breadcrumbs';
import DefinitionList from '../../../../../../old-code/components/DefinitionList/DefinitionList';
import { ReleaseRuleReadonlyPageDockerPatches } from './__dockerPatches/ReleaseRuleReadonlyPageDockerPatches';
import { ReleaseRuleReadonlyPageSandboxPatches } from './__sandboxPatches/ReleaseRuleReadonlyPageSandboxPatches';

import classes from './ReleaseRuleReadonlyPage.module.scss';

interface Props {
   projectId: string;
   stageId: string;
   rule: ReleaseRule;

   onEdit(): void;

   onRemove(): void;
}

export const ReleaseRuleReadonlyPage: React.FC<Props> = React.memo(({ stageId, projectId, rule, onEdit, onRemove }) => {
   const links = useMemo(
      () =>
         projectId
            ? [
                 { name: 'All', url: urlBuilder.projects() },
                 { name: projectId, url: urlBuilder.project(projectId) },
                 { name: stageId, url: urlBuilder.stage(stageId) },
                 { name: rule.id, url: urlBuilder.releaseRulePage(stageId, rule.id) },
              ]
            : [],
      [projectId, rule.id, stageId],
   );

   const header = (
      <div className={classes.header}>
         <div className={classes.title}>
            <div className={classes.breadcrumbs}>
               <Breadcrumbs links={links} />
            </div>
            <div className={classes.subtitle}>{rule.type} release rule</div>
            <h1>{rule.id}</h1>
         </div>
         <div className={classes.actions}>
            <Button onClick={onEdit}>
               <FontAwesomeIcon icon={faPencil} />
            </Button>
            <Button onClick={onRemove}>
               <FontAwesomeIcon icon={faTrashAlt} />
            </Button>
         </div>
      </div>
   );

   const sandboxParams = rule.form?.sandbox;
   const dockerParams = rule.form?.docker;

   const resourceParams = sandboxParams
      ? [
           {
              name: 'Task type',
              value: rule?.form?.sandbox?.taskType ?? EMPTY_VALUE,
           },
           {
              name: 'Resource types',
              value: isEmpty(rule?.form?.sandbox?.resources) ? EMPTY_VALUE : rule!.form!.sandbox!.resources.join(', '),
           },
           {
              name: 'Resource attributes',
              value: isEmpty(sandboxParams.attributes)
                 ? EMPTY_VALUE
                 : sandboxParams.attributes.map(v => `${v.key}=${v.value}`).join('; '),
           },
        ]
      : [];

   return (
      <Page header={header} className={classes.page}>
         <DefinitionList
            className={classes.list}
            items={[
               {
                  name: 'Description',
                  value: rule.description ? rule.description : EMPTY_VALUE,
               },
               {
                  name: 'Release types',
                  value: isEmpty(rule.form!.releaseTypes) ? 'any' : rule.form!.releaseTypes.join(', '),
               },
               {
                  name: 'Autocommit tickets',
                  value: rule.autocommit ? 'Yes' : 'No',
               },
               {
                  name: 'Release rule type',
                  value: rule.type,
               },
               ...resourceParams,
            ]}
         />

         <h2 className={classes.patchesTitle}>Patches</h2>

         {rule.type === ReleaseRuleType.Sandbox ? (
            <ReleaseRuleReadonlyPageSandboxPatches patches={sandboxParams?.patches} />
         ) : (
            <ReleaseRuleReadonlyPageDockerPatches patches={dockerParams?.patches} />
         )}

         <DevJson summary={'Rule'}>{rule}</DevJson>
      </Page>
   );
});

ReleaseRuleReadonlyPage.displayName = 'ReleaseRuleReadonlyPage';
