import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import {
   EMPTY_VALUE,
   ExtendedFieldConfig,
   ExternalLink,
   FieldLayout2,
   FormButton,
   InputField2,
   isEmpty,
   isEqual,
   RemoveButton,
   unique,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useEffect } from 'react';
import { StagePart } from '../../../../../../components';
import { KeyValueField } from '../../../../../../components/forms';
import { EXTERNAL_LINKS } from '../../../../../../models';

import { defaultSandboxPatch, ResourceType, SandboxFormParams, SandboxPatch } from '../../../../../../models/ui';

import { getSequentialId } from '../../../../../../utils';
import { TargetSelect } from './components/TargetSelect';

import classes from './SandboxParamsField.module.css';

interface PatchesTableRow {
   resourceType: ReactNode;
   target: ReactNode;
   patchName: ReactNode;
   buttons: ReactNode;
}

const patchesColumns: TableColumnConfig<PatchesTableRow>[] = [
   { id: 'resourceType', name: 'Resource type' },
   { id: 'target', name: 'Target' },
   { id: 'patchName', name: 'Patch name' },
   { id: 'buttons', name: '' },
];

export const SandboxParamsField = React.memo((props: ExtendedFieldConfig<any, SandboxFormParams | null>) => {
   const { field, onChange, disabled } = useExtendedField(props);
   const { value, name } = field;
   const { required } = props;

   useEffect(() => {
      if (value) {
         const resources = unique((value!.patches ?? []).map(p => p.resourceType));

         // Это можно было бы делать один раз при обновлении правила, но списки типов ресурсов патчей и самого правила
         // могут отличаться (см. DEPLOY-5171) и в этом случае пользователь не увидит в интерфейсе, если у него что-то в
         // этом месте (не) поменялось (DEPLOY-5165)
         if (!isEqual(resources, value.resources)) {
            onChange({
               ...value,
               resources,
            });
         }
      }
   }, [onChange, value]);

   const handlePatchesChange = useCallback(
      (patches: SandboxPatch[]) => {
         if (value) {
            onChange({
               ...value,
               patches,
               resources: unique(patches.map(p => p.resourceType)),
            });
         }
      },
      [onChange, value],
   );

   const handleAddPatch = useCallback(() => {
      handlePatchesChange([...(value?.patches ?? []), { ...defaultSandboxPatch }]);
   }, [handlePatchesChange, value]);

   const handleDeletePatch = useCallback(
      (i: number) => {
         const patches = [...(value?.patches ?? [])];

         patches.splice(i, 1);

         handlePatchesChange(patches);
      },
      [handlePatchesChange, value],
   );

   const handlePatchChange = useCallback(
      (i: number, newPatch: SandboxPatch) => {
         handlePatchesChange((value?.patches ?? []).map((p, index) => (i === index ? newPatch : p)));
      },
      [handlePatchesChange, value],
   );

   const handlePatchTargetSelect = useCallback(
      (i: number, v: StagePart | null) => {
         const patch = value?.patches[i];

         if (patch) {
            const siblingIds = (value?.patches ?? []).filter(p => p.id !== patch.id).map(p => p.id);
            const defaultId = v?.resource?.id
               ? getSequentialId(`patch-${v.resource.id}`, siblingIds)
               : getSequentialId(`patch`, siblingIds);

            const newPatch: SandboxPatch = {
               ...patch,
               deployUnitId: v?.deployUnit ?? '',
               ref: v?.resource?.id ?? '',
               type: v?.resource?.type ?? ResourceType.Layer,
               id: patch.id && !isEmpty(patch.id) ? patch.id : defaultId,
            };

            if (v?.resource?.type === ResourceType.DynamicResource) {
               handlePatchChange(i, {
                  ...newPatch,
                  deployGroupMark: v.resource!.deployGroupMark ?? '',
               });
            } else {
               handlePatchChange(i, newPatch);
            }
         }
      },
      [handlePatchChange, value],
   );

   return (
      <FieldLayout2 {...props} required={false}>
         <InputField2
            name={`${name}.taskType`}
            label={'Sandbox task type'}
            required={required}
            disabled={disabled}
            controlProps={{ placeholder: 'Enter sandbox task type' }}
         />

         <KeyValueField
            name={`${name}.attributes`}
            label={'Resource attributes'}
            disabled={disabled}
            controlProps={{ addButton: 'Add attribute' }}
            hint={
               <>
                  Match resources only with these{' '}
                  <ExternalLink href={EXTERNAL_LINKS.docs.sandboxResourceAttributes}>attributes</ExternalLink>
               </>
            }
         />

         <FieldLayout2 name={`${name}.resources`} label={'Resource types'} readonly={true} readonlyDots={true}>
            <div className={classes.sandboxResourceTypes}>
               {value?.resources && !isEmpty(value.resources)
                  ? value.resources.map(v => (!isEmpty(v) ? v : EMPTY_VALUE)).join(', ')
                  : EMPTY_VALUE}
            </div>
         </FieldLayout2>

         <FieldLayout2
            name={`${field.name}.patches`}
            label={'Deploy patches'}
            disabled={disabled}
            required={required}
            // выводим общую ошибку массива только, когда массив пустой
            // (например, "You should add at least one patch")
            // если массив не пустой, то выводим ошибки в отдельных полях
            hideErrors={value?.patches && !isEmpty(value.patches)}
            data-test={'SandboxParams:DeployPatches'}
         >
            <Table
               columns={patchesColumns}
               data={(value?.patches ?? []).map((patch, i) => ({
                  resourceType: (
                     <div className={classes.resourceType} data-e2e={`DeployPatches:${i + 1}:ResourceType`}>
                        <InputField2
                           name={`${field.name}.patches[${i}].resourceType`}
                           label={null}
                           disabled={disabled}
                           controlProps={{ placeholder: 'Enter resource type' }}
                        />
                     </div>
                  ),
                  target: (
                     <div className={classes.target} data-e2e={`DeployPatches:${i + 1}:ResourceTarget`}>
                        <FieldLayout2 name={`${field.name}.patches[${i}].ref`} label={null}>
                           <TargetSelect
                              type={patch.type}
                              id={patch.ref}
                              deployUnit={patch.deployUnitId}
                              deployGroupMark={patch.deployGroupMark}
                              onSelect={(v: StagePart | null) => handlePatchTargetSelect(i, v)}
                              disabled={disabled}
                           />
                        </FieldLayout2>
                     </div>
                  ),
                  patchName: (
                     <div className={classes.patchName} data-e2e={`DeployPatches:${i + 1}:Name`}>
                        <InputField2
                           name={`${field.name}.patches[${i}].id`}
                           label={null}
                           disabled={disabled}
                           controlProps={{ placeholder: 'Enter patch name' }}
                        />
                     </div>
                  ),
                  buttons: (
                     <div data-e2e={`DeployPatches:${i + 1}:RemovePatch`}>
                        <RemoveButton onClick={() => handleDeletePatch(i)} />
                     </div>
                  ),
               }))}
               emptyMessage={'No patches. You should add at least one.'}
               verticalAlign={'top'}
            />

            {/* TODO: AddButton #DEPLOY-4915 */}
            <div data-e2e={'DeployPatches:AddPatch'}>
               <FormButton onClick={handleAddPatch} skipLeftSpace={true} icon={faPlus}>
                  Add patch
               </FormButton>
            </div>
         </FieldLayout2>
      </FieldLayout2>
   );
});

SandboxParamsField.displayName = 'SandboxParamsField';
