import { capitalize, fromQuery, toasts, toQuery, useDismounted, useTitle } from '@yandex-infracloud-ui/libs';
import { FormikHelpers, FormikProps } from 'formik';
import React, { useCallback, useContext, useEffect, useMemo, useRef } from 'react';
import { RouteComponentProps } from 'react-router';
import { finalize, takeUntil } from 'rxjs/operators';

import { FormPage } from '../../../../components';
import { handleApiError, NotificationsContext, urlBuilder } from '../../../../models';
import {
   DefaultReleaseRuleType,
   ReleaseRuleFormParams as FormParams,
   releaseRuleInitialFormParams as initialValues,
   ReleaseRuleType,
} from '../../../../models/ui/releases';
import { useStage } from '../../../../redux';
import { ypApi } from '../../../../services';

import { validationSchema } from './components/ReleaseRuleForm/models';
import { ReleaseRuleForm } from './components/ReleaseRuleForm/ReleaseRuleForm';
import { useReleaseRuleFormContext } from './models';

/**
 * Страница с новым релизным правилом.
 *
 * По-умолчанию открывается для Sandbox
 * Если передать ?type=docker, то будет выбран Docker
 */
export const NewReleaseRulePage: React.FC<RouteComponentProps<{ stageId: string; type: string }>> = React.memo(
   ({ match, history, location }) => {
      const { stageId } = match.params;
      const { stage } = useStage(stageId);
      const typeFromQuery = getTypeFromQuery(location.search);
      const title = `New ${capitalize(typeFromQuery)} rule / ${stageId}`;
      const notifications = useContext(NotificationsContext);

      // region hooks
      const dismounted = useDismounted();

      const initialValuesPatched = useMemo(
         () => ({
            ...initialValues,
            id: `${stageId}-rule`,
            type: typeFromQuery,
         }),
         [stageId, typeFromQuery],
      );

      // Проброс данных из stage внутрь кастомных полей формы
      const formContext = useReleaseRuleFormContext(stageId, true);
      const formRef = useRef<FormikProps<FormParams>>();
      // endregion

      // region effects
      useTitle(title);

      useEffect(() => {
         // Обновление типа правила в форме при обновлении URL
         if (formRef.current) {
            formRef.current.setFieldValue('type', typeFromQuery);
         }
      }, [typeFromQuery]);
      // endregion

      // region handlers
      const goToReleaseList = useCallback(() => history.push(urlBuilder.stageDeployTickets(stageId)), [
         history,
         stageId,
      ]);

      const handleSubmit = useCallback(
         (v: FormParams, helpers: FormikHelpers<FormParams>) => {
            ypApi
               .createReleaseRule(stageId, v)
               .pipe(
                  finalize(() => helpers.setSubmitting(false)),
                  takeUntil(dismounted),
               )
               .subscribe(() => {
                  toasts.success(`Release rule ${v.id} created`);
                  goToReleaseList();
               }, handleApiError('Creating release rule', notifications));
         },
         [dismounted, goToReleaseList, notifications, stageId],
      );

      const handleFieldChange = useCallback(
         (field: string, v: any) => {
            if (field === 'type') {
               const t = v as ReleaseRuleType;
               const search = { type: t === DefaultReleaseRuleType ? undefined : t };

               history.push({ search: toQuery(search) });
            }
         },
         [history],
      );
      // endregion

      // region render
      return (
         <FormPage
            cancelButtonText={'Cancel'}
            subTitle={'New release rule'}
            submitButtonText={'Create'}
            title={`${title} for stage "${stageId}"`}
            formRef={formRef}
            loading={stage === null ? `Stage "${stageId}" is loading` : null}
            initialValues={initialValuesPatched}
            validationSchema={validationSchema}
            formContext={formContext}
            onFieldChange={handleFieldChange as any} // TODO typings
            onSubmit={handleSubmit}
            onCancel={goToReleaseList}
         >
            {() => <ReleaseRuleForm />}
         </FormPage>
      );
      // endregion
   },
);

NewReleaseRulePage.displayName = 'NewReleaseRulePage';

/**
 * Парсит QueryString и возвращает оттуда правильный ReleaseRuleType
 * Если передано что-то неподдерживаемое, то используется дефолтный (Sandbox)
 *
 * @param search Строка после ?
 */
function getTypeFromQuery(search: string): ReleaseRuleType {
   const supportedValues = Object.values(ReleaseRuleType);
   const parsed = fromQuery(search) as { type: ReleaseRuleType };
   if (supportedValues.includes(parsed.type)) {
      return parsed.type;
   }

   return DefaultReleaseRuleType;
}
