import { useTitle } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useContext, useEffect, useMemo, useState } from 'react';
import { useDispatch } from 'react-redux';
import { RouteComponentProps, useHistory } from 'react-router';

import { urlBuilder } from '../../../../models';
import { DeepNonNullable } from '../../../../models/api';
import { StageRequestKeys } from '../../../../models/ui';
import { TProject } from '../../../../proto-typings';
import { getProjects, useNetworkErrors, useStage } from '../../../../redux';
import { useConfig } from '../../../../services';
import { DeployUnitBlock } from './components/DeployUnitBlock/DeployUnitBlock';
import { DeployUnitList } from './components/DeployUnitList/DeployUnitList';
import { StageHeader } from './components/StageHeader/StageHeader';
import { StageInfoBlock } from './components/StageInfoBlock/StageInfoBlock';
import { StageTab } from './components/StageTab/StageTab';
import { StatusTab } from './components/StatusTab/StatusTab';
import {
   ClusterContext,
   getEmptyStatusViewStore,
   StatusViewContext,
   StatusViewStoreUpdater,
   StatusViewUpdateContext,
} from './context';
import { useStatusViewUpdate } from './hooks';

import classes from './StageStatusPage.module.css';

const UNAVAILABLE_MAX_DU_NAME = '~';
const STAGE_INFO_PATH = '~stage-info';

// url - источник истины для состояния

export const StageStatusPage: React.FC<
   RouteComponentProps<{ stageId: string; deployUnitId?: string; clusterId?: string }>
> = React.memo(({ match }) => {
   const { stageId, deployUnitId, clusterId } = match.params;
   const isStageInfo = deployUnitId === STAGE_INFO_PATH;
   const requestKey = StageRequestKeys.getOne({ id: stageId });

   const { rawStage, stageStatusLabel, stageStatus } = useStage(stageId);

   // проверяем, что загрузилась вся спека, включая какое-нибудь надёжное поле в спеке,
   // кроме деплой юнитов, которые могут быть пустыми (как в спеке, так и в статусе)
   const isFullData = rawStage && rawStage.meta && rawStage.spec?.revision && rawStage.status;

   const projectId = rawStage?.meta?.project_id;

   const paths = useCallback((p: DeepNonNullable<TProject>) => [p.spec.account_id, p.spec.monitoring_project], []);
   const dispatch = useDispatch();

   const getProject = useCallback((id: string) => dispatch(getProjects({ ids: [id], paths })), [dispatch, paths]);

   useEffect(() => {
      if (projectId) {
         getProject(projectId);
      }
   }, [getProject, projectId]);

   useTitle(`Status / ${stageId}`);

   const { deployUnitLabels } = stageStatusLabel ?? {};
   const duList = Array.from(deployUnitLabels?.keys() ?? []);

   const firstDeployUnit = duList.reduce((first, name) => (first < name ? first : name), UNAVAILABLE_MAX_DU_NAME);
   const activeDeployUnitId = deployUnitId ?? firstDeployUnit;

   const { locations: duLocations, replicaSetsInfo } = stageStatus?.deployUnits?.get(activeDeployUnitId) ?? {};
   const { clusters } = useConfig()!;
   const firstCluster = clusters.map(e => e.value).find(e => duLocations?.has(e));

   const activeClusterId = clusterId ?? firstCluster;

   const history = useHistory();

   const onChangeDuId = useCallback(
      (duId: string) => {
         history.push(urlBuilder.stageStatus(stageId, duId));
      },
      [history, stageId],
   );

   const networkErrors = useNetworkErrors([requestKey]);
   const stageError = networkErrors[requestKey];

   // update parent StatusViewStore
   const updateParentStatusViewStore = useContext(StatusViewUpdateContext);

   // local StatusViewStore
   const [statusViewStore, setStatusViewStore] = useState(getEmptyStatusViewStore());

   const updateStatusViewStore: StatusViewStoreUpdater = useCallback(
      update => {
         setStatusViewStore(update);
         updateParentStatusViewStore(update);
      },
      [updateParentStatusViewStore],
   );

   const replicaSetKey = useMemo(
      () =>
         Object.entries(replicaSetsInfo ?? {}).find(([key, rs]) => rs.locations.has(activeClusterId ?? ''))?.[0] ?? '',
      [activeClusterId, replicaSetsInfo],
   );

   const activeClusters = useMemo(() => new Set(activeClusterId ? [activeClusterId] : []), [activeClusterId]);

   // update statusView models, see README.md
   useStatusViewUpdate({
      stageId,
      duId: activeDeployUnitId,
      replicaSetKey,
      activeClusters,
      statusViewStore,
      updateStatusViewStore,
   });

   if (!isFullData || !deployUnitLabels) {
      return null;
   }

   const duSize = deployUnitLabels.size;
   const statusData = (
      <div className={classes.statusData}>
         <div className={classes.duTabs}>
            <StatusTab
               onSelect={() => {
                  history.push(urlBuilder.stageStatus(stageId, STAGE_INFO_PATH));
               }}
               active={isStageInfo}
               dataTest={'status--stage'}
               ariaLabel={'Stage status'}
            >
               <StageTab stageId={stageId} />
            </StatusTab>
            <div className={classes.duCount}>
               Deploy units
               <div className={classes.count}>{duSize}</div>
            </div>
            <DeployUnitList stageId={stageId} onSelect={onChangeDuId} selectedDuId={activeDeployUnitId} />
         </div>
         <div className={classes.statusInfo}>
            {isStageInfo ? (
               <StageInfoBlock stageId={stageId} />
            ) : deployUnitLabels.has(activeDeployUnitId) ? (
               <ClusterContext.Provider value={activeClusterId ?? null}>
                  <DeployUnitBlock stageId={stageId} duId={activeDeployUnitId} projectId={projectId} />
               </ClusterContext.Provider>
            ) : (
               <div>no deploy unit with the name {activeDeployUnitId}</div>
            )}
         </div>
      </div>
   );

   return (
      <div
         className={`${(stageError?.request?.lastOkTimestamp ?? 0) > 0 ? classes.old : ''}`}
         data-test={'view-stage--status'}
      >
         <StatusViewContext.Provider value={statusViewStore}>
            <StatusViewUpdateContext.Provider value={setStatusViewStore}>
               <StageHeader stageId={stageId} />
               {statusData}
            </StatusViewUpdateContext.Provider>
         </StatusViewContext.Provider>
      </div>
   );
});

StageStatusPage.displayName = 'StageStatusPage';
