import { faExclamationTriangle } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import { ButtonLink, classNames, ExternalLink, Loader, modalService, useDismounted } from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useMemo } from 'react';
import { useSelector } from 'react-redux';
import { takeUntil } from 'rxjs/operators';

import { EXTERNAL_LINKS, urlBuilder } from '../../../../../../models';
import {
   DeployUnit,
   DeployUnitType,
   extractDeployUnitClusterErrors,
   getDuWarningsCount,
   PodConverter,
} from '../../../../../../models/ui';
import { getIssueSummaryFromList, IssueLevelColors } from '../../../../../../modules/infra-doctor';
import {
   RootState,
   selectDuPodIdList,
   selectProject,
   useDeployUnit,
   useStagePermissions,
   useStageProblems,
} from '../../../../../../redux';
import { pluralNumber, restoreObjectFromKey } from '../../../../../../utils';
import { usePodStatusView, useReplicaSetStatusView } from '../../hooks';
import { DeployUnitErrors } from '../DeployUnitErrors/DeployUnitErrors';
import { PerLocationInfo } from '../PerLocationInfo/PerLocationInfo';
import { ReplicaSets } from '../ReplicaSets/ReplicaSets';
import { RuntimeErrorModal } from '../RuntimeErrorModal/RuntimeErrorModal';

import classes from './DeployUnitBlock.module.css';

const duTypeCaption: Record<DeployUnitType, ReactNode> = {
   [DeployUnitType.PerCluster]: 'Per cluster replica sets',
   [DeployUnitType.MultiCluster]: 'Multi cluster replica set',
};

const pluralPods = (pods: number) => pluralNumber(pods, 'pod', 'pods');

const unavailablePodsCaption: Record<DeployUnitType, (du: DeployUnit) => ReactNode> = {
   [DeployUnitType.MultiCluster]: deployUnit => pluralPods(deployUnit.disruptionBudget ?? 0),
   [DeployUnitType.PerCluster]: deployUnit =>
      Object.keys(deployUnit.locations)
         .sort()
         .map(cluster => {
            const podCount = deployUnit.locations[cluster].disruptionBudget;
            return `${cluster.toLocaleUpperCase()}: ${pluralPods(podCount ?? 0)}`;
         })
         .join(', '),
};

interface Props {
   stageId: string;
   duId: string;
   projectId?: string;
}

export const DeployUnitBlock: React.FC<Props> = React.memo(({ stageId, duId, projectId }) => {
   const userHasAccessToEdit = useStagePermissions(stageId);

   const { deployUnitStatus, deployUnit: deployUnitSpec } = useDeployUnit(stageId, duId);

   const currentProject = useSelector(projectId ? (s: RootState) => selectProject(s, projectId) : () => null);
   const monitoringProject = currentProject?.spec?.monitoring_project;
   const isAlertingEnabled = deployUnitSpec?.settings.alerting.state;

   const { replicaSetsInfo, locations, podsCount, currentTarget: deployUnit } = deployUnitStatus ?? {};
   const replicaSetSelectData = useMemo(
      () =>
         Object.keys(replicaSetsInfo ?? {}).map(restoreObjectFromKey) as {
            location: string;
            id: string;
         }[],
      [replicaSetsInfo],
   );

   const { replicaSetStatusViewList } = useReplicaSetStatusView({ replicaSetSelectData });

   const podsSelector = useCallback(
      (state: RootState) => selectDuPodIdList(state, PodConverter.getPodSetId(stageId, duId)),
      [duId, stageId],
   );
   const podSelectData = useSelector(podsSelector);

   const { podStatusViewList } = usePodStatusView({ podSelectData });

   // зависимость ошибок от подгрузок дочерних компонентов
   // TODO: вынести загрузку наверх
   const warnings = extractDeployUnitClusterErrors({
      replicaSets: replicaSetStatusViewList,
      pods: podStatusViewList.map(podStatusView => ({ pod: podStatusView, cluster: podStatusView.location })),
   });
   const warningCount = getDuWarningsCount(warnings);

   const dismounted = useDismounted();
   const openErrorModal = useCallback(() => {
      modalService
         .open(RuntimeErrorModal, {
            title: 'Deploy Unit',
            id: duId,
            warningCount,
            errors: (<DeployUnitErrors warnings={warnings} />) as any,
         })
         .pipe(takeUntil(dismounted))
         .subscribe();
   }, [dismounted, duId, warningCount, warnings]);

   const { issuesByDeployUnit } = useStageProblems(stageId);
   const { summary: duProblemSummary } = useMemo(
      () => getIssueSummaryFromList({ issues: issuesByDeployUnit[duId] ?? [] }),
      [duId, issuesByDeployUnit],
   );

   let content: ReactNode;

   if (!(deployUnitStatus && deployUnitSpec && locations && podsCount)) {
      content = null;
   } else if (locations.size === 0) {
      content = (
         <p>
            No pods in <strong>{duId}</strong>
         </p>
      );
   } else {
      content = (
         <>
            <div>
               <div className={classes.header}>
                  <div className={classes.podsCount}>
                     <div>
                        Pods
                        {podsCount.total ? (
                           <span className={classes.count}> {podsCount.total}</span>
                        ) : (
                           <Loader inline={true} />
                        )}
                     </div>
                     {deployUnit && (
                        <div className={classes.unavailable}>
                           {duTypeCaption[deployUnit.type]}, {unavailablePodsCaption[deployUnit.type](deployUnitSpec)}{' '}
                           can be unavailable.
                        </div>
                     )}
                  </div>
               </div>

               <PerLocationInfo stageId={stageId} duId={duId} />
               <ReplicaSets stageId={stageId} duId={duId} />
            </div>
         </>
      );
   }

   const errors = (
      <div>
         <FontAwesomeIcon icon={faExclamationTriangle} /> Show {warningCount} error{warningCount === 1 ? '' : 's'}
      </div>
   );

   return (
      <div>
         <div className={classes.duHeader}>
            <h2>{duId}</h2>
            {userHasAccessToEdit.edit && <ButtonLink to={urlBuilder.stageEdit(stageId, { duId })}>Edit</ButtonLink>}
            <ButtonLink to={urlBuilder.stageLogs(stageId, { deployUnitId: duId })}>Logs</ButtonLink>
            <ButtonLink to={urlBuilder.stageStatusInfo(stageId, { params: { deployUnitIds: new Set([duId]) } })}>
               Health:{' '}
               <span style={{ color: IssueLevelColors[duProblemSummary.summaryLevel] }}>
                  {duProblemSummary.summaryLevel}
               </span>
            </ButtonLink>
            {isAlertingEnabled && monitoringProject && (
               <ExternalLink href={EXTERNAL_LINKS.alertsList(duId, stageId, monitoringProject)}>Alerts</ExternalLink>
            )}
            {warningCount > 0 ? (
               <Button
                  onClick={openErrorModal}
                  view={'flat'}
                  className={classNames({ [classes.existErrors]: warningCount > 0 })}
               >
                  {errors}
               </Button>
            ) : (
               <span className={classes.noErrors}>No errors</span>
            )}
         </div>
         {content}
      </div>
   );
});

DeployUnitBlock.displayName = 'DeployUnitBlock';
