import React, { ReactNode } from 'react';

import { Collapse } from '@yandex-data-ui/common';

import { DuWarningResult } from '../../../../../../models/ui/runtimeDeploy';
import { PodError } from '../PodError/PodError';
import { restoreObjectFromKey } from '../../../../../../utils';
import { ReplicaSetError } from '../ReplicaSetError/ReplicaSetError';
import { YpErrorHeader } from '../../../../../../components';

import classes from './DeployUnitErrors.module.css';
import { PodStatus } from '../PodStatus/PodStatus';
import { RuntimeObjectRevision } from '../RuntimeObjectRevision/RuntimeObjectRevision';

interface YpObjectErrorBlockProps {
   id: string;
   title: string;
   location: string;
   count: number;
   additionalData?: ReactNode;
   expand?: boolean;
}

export const YpObjectErrorBlock: React.FC<YpObjectErrorBlockProps> = ({
   id,
   location,
   title,
   count,
   additionalData,
   expand = true,
   children,
}) => {
   if (count === 0) {
      return null;
   }
   return (
      <div key={`${id}-${location}`}>
         <Collapse
            defaultIsExpand={expand}
            title={
               <YpErrorHeader title={title} id={id} location={location} count={count}>
                  {additionalData}
               </YpErrorHeader>
            }
         >
            {children}
         </Collapse>
      </div>
   );
};

interface Props {
   warnings: DuWarningResult;
}

export const DeployUnitErrors: React.FC<Props> = React.memo(({ warnings }) => (
   <div>
      <p>These are all errors of a selected deploy unit</p>
      {Array.from(warnings.replicaSets.entries()).map(([replicaSetKey, replicaSetData]) => {
         const { id, location } = restoreObjectFromKey(replicaSetKey);
         return (
            <div className={classes.errorBlock}>
               <YpObjectErrorBlock title={'Replica set'} id={id} location={location} count={replicaSetData.count}>
                  <ReplicaSetError warning={replicaSetData} />
               </YpObjectErrorBlock>
            </div>
         );
      })}
      {Array.from(warnings.pods.entries()).map(([podKey, podData], i) => {
         const { id, cluster } = restoreObjectFromKey(podKey);
         return (
            <div className={classes.errorBlock}>
               <YpObjectErrorBlock
                  title={'Pod'}
                  id={id}
                  location={cluster}
                  count={podData.totalWarningCount}
                  expand={i === 0}
                  additionalData={
                     <div className={classes.podData}>
                        <PodStatus pod={podData} inline={true} />
                        <RuntimeObjectRevision
                           revision={podData.revision}
                           targetRevision={podData.targetRevision}
                           progress={podData.statusStateInfo.inProgress.active}
                           full={true}
                        />
                     </div>
                  }
               >
                  <PodError pod={podData} />
               </YpObjectErrorBlock>
            </div>
         );
      })}
   </div>
));

DeployUnitErrors.displayName = 'DeployUnitErrors';
