import React from 'react';

import { classNames, isEqual } from '@yandex-infracloud-ui/libs';

import { ApprovalList } from '../../../../../../components';
import { OrderList } from '../../../../../../components/lib';
import { DeployUnitType, PerLocationStrategy } from '../../../../../../models/ui';

import { useDeployUnit } from '../../../../../../redux';
import { parseYpDatetime } from '../../../../../../utils';

import classes from './PerLocationInfo.module.css';

interface Props {
   stageId: string;
   duId: string;
}

export const PerLocationInfo: React.FC<Props> = React.memo(({ stageId, duId }) => {
   const { deployUnit, deployUnitStatus } = useDeployUnit(stageId, duId);

   if (!deployUnit) {
      return null;
   }

   if (deployUnit.type !== DeployUnitType.PerCluster) {
      return null;
   }

   const { perLocationSettings } = deployUnit;
   const { overrides } = deployUnitStatus ?? {};

   const {
      strategy: initialStrategy,
      locationOrder: initialLocationOrder,
      isCustom: initialUseCustom,
   } = perLocationSettings;

   const { strategy, locationOrder, isCustom } = {
      ...perLocationSettings,
      ...(overrides?.deploySettings?.perLocationSettings ?? {}),
   };

   if (!isCustom) {
      return null;
   }

   const perLocationStrategyChanged = strategy !== initialStrategy;
   const orderChanged = !isEqual(locationOrder, initialLocationOrder);

   const { login, time } = overrides?.deploySettings?.meta ?? {};
   const separator = strategy === PerLocationStrategy.Parallel ? ', ' : null;

   return (
      <div>
         Deploy per location: {strategy}
         <div className={classNames(classes.list, classes.old)}>
            <OrderList items={locationOrder.map(e => e.toLocaleUpperCase())} separator={separator} />
         </div>
         {(orderChanged || perLocationStrategyChanged) && (
            <>
               {login && (
                  <div className={classes.list}>
                     <ApprovalList
                        approvals={[
                           {
                              login,
                              status: 'approved',
                              timestamp: time ? parseYpDatetime(time) : null,
                           },
                        ]}
                     />
                  </div>
               )}
               {initialUseCustom && (
                  <div>
                     Spec value: {perLocationStrategyChanged && initialStrategy}
                     <div className={classNames(classes.list, classes.old)}>
                        <OrderList items={initialLocationOrder.map(e => e.toLocaleUpperCase())} separator={separator} />
                     </div>
                  </div>
               )}
            </>
         )}
      </div>
   );
});

PerLocationInfo.displayName = 'PerLocationInfo';
