import { Button, Checkbox, HelpPopover, TextInput } from '@yandex-cloud/uikit';
import { YCSelect, YCSelectItem } from '@yandex-data-ui/common';
import { ExternalLink } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useState } from 'react';
import { EXTERNAL_LINKS } from '../../../../../../models';

import {
   getQueryFromMainFilters,
   PodFilters,
   PodRevisionGroup,
   podRevisionGroupData,
   podRevisionGroupList,
   PodStatusGroup,
   podStatusGroupData,
   podStatusGroupList,
} from '../../model';

import classes from './PodFiltersBlock.module.css';

const statusGroupItems: YCSelectItem[] = podStatusGroupList.map(group => {
   const { title } = podStatusGroupData[group];
   return {
      value: group,
      title,
   };
});

const getRevisionGroupItems: (revision: number) => YCSelectItem[] = revision =>
   podRevisionGroupList.map(group => {
      const { title } = podRevisionGroupData(revision)[group];
      return {
         value: group,
         title,
      };
   });

interface Props {
   value: PodFilters;
   revision: number;

   onChange(filters: PodFilters): void;
}

export const PodFiltersBlock: React.FC<Props> = React.memo(({ value, onChange, revision }) => {
   const { statusGroup, revisionGroup, isCustomFilter, customFilter } = value;

   // частый ввод в текстовом поле не должен триггерить запросы
   const [customText, setCustomText] = useState(customFilter);

   const onFieldChange = useCallback(
      <K extends keyof PodFilters>(name: K, fieldValue: PodFilters[K], customValue?: PodFilters) => {
         const oldFilters = customValue ?? value;
         const newFilters = { ...oldFilters };
         newFilters[name] = fieldValue;
         onChange(newFilters);

         return newFilters;
      },
      [onChange, value],
   );

   return (
      <div className={classes.filters} data-test={'pods-filter'}>
         <div className={classes.mainFilters}>
            <div data-test={'pods-filter--status'}>
               <YCSelect
                  value={statusGroup}
                  items={statusGroupItems}
                  placeholder={'Select pod status'}
                  disabled={isCustomFilter}
                  showSearch={false}
                  onUpdate={group => onFieldChange('statusGroup', group as PodStatusGroup)}
               />
            </div>
            <div data-test={'pods-filter--revision'}>
               <YCSelect
                  value={revisionGroup}
                  items={getRevisionGroupItems(revision)}
                  placeholder={'Select pod revision'}
                  disabled={isCustomFilter}
                  showSearch={false}
                  onUpdate={group => onFieldChange('revisionGroup', group as PodRevisionGroup)}
               />
            </div>
            <div data-test={'pods-filter--custom-checkbox'} className={classes.custom}>
               <Checkbox
                  checked={isCustomFilter}
                  onUpdate={checked => {
                     const withCustom = onFieldChange('isCustomFilter', checked);
                     const prefferedCustom = getQueryFromMainFilters(value, revision);
                     if (prefferedCustom) {
                        onFieldChange('customFilter', prefferedCustom, withCustom);
                        setCustomText(prefferedCustom);
                     }
                  }}
               >
                  Custom query string
               </Checkbox>{' '}
               <HelpPopover
                  content={
                     <>
                        Operators "AND" and "OR" may be used to combine more than one condition. Examples:
                        <ul>
                           <li>
                              <code>[/meta/creation_time] {'>'} 875016084755063</code>
                           </li>
                           <li>
                              <code>
                                 ([/status/agent/pod_agent_payload/status/ready/status]="true") AND (NOT
                                 [/status/agent/pod_agent_payload/status/revision]=28u)
                              </code>
                           </li>
                        </ul>
                        <br />
                        See <ExternalLink href={EXTERNAL_LINKS.wiki.ypApi}>documentation</ExternalLink> for details.
                     </>
                  }
                  openOnHover={false}
               />
            </div>
         </div>
         {isCustomFilter && (
            <div className={classes.customFilter} data-test={'pods-filter--custom'}>
               <div data-test={'pods-filter--custom-input'}>
                  <TextInput
                     view={'normal'}
                     size={'m'}
                     value={customText}
                     placeholder={'Query example: [/labels/node/cpu_model] = "cpu_model"'}
                     hasClear={true}
                     onKeyDown={e => {
                        if (e.keyCode === 13) {
                           onFieldChange('customFilter', customText);
                        }
                     }}
                     onUpdate={text => setCustomText(text)}
                  />
               </div>
               <div data-test={'pods-filter--custom-button'}>
                  <Button
                     onClick={() => {
                        onFieldChange('customFilter', customText);
                     }}
                  >
                     search
                  </Button>
               </div>
            </div>
         )}
      </div>
   );
});

PodFiltersBlock.displayName = 'PodFilters';
