import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { Checkbox } from '@yandex-cloud/uikit';
import {
   classNames,
   isEqual,
   Loader,
   Pagination,
   TIMES_IN_MS,
   UpdateTimerMode,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs';

import { useDispatch, useSelector } from 'react-redux';

import { useHistory, useLocation } from 'react-router';
import { YpErrorTooltip } from '../../../../../../components/network';
import { urlBuilder } from '../../../../../../models';
import { YpLocation } from '../../../../../../models/api';
import { PodConverter, ReplicaSetType } from '../../../../../../models/ui';
import {
   fetchPods,
   RootState,
   selectPods,
   useDeployUnit,
   useNetworkErrors,
   useNetworkRequests,
   useReplicaSet,
   useRequestControl,
} from '../../../../../../redux';
import { RequestState } from '../../../../../../redux/slices/network/model';
import { createKey, useRafDelay, useSkipAfterFirstRender } from '../../../../../../utils';
import {
   defaultPodFilters,
   getFetchQuery,
   getFiltersFromUrl,
   getPaginationFromUrl,
   getUrlParamsFromPodData,
   PodFilters,
   statusSettingsStorage,
   podsPaginationPerPageKey,
   podsPerPages,
} from '../../model';
import { PodFiltersBlock } from '../PodFiltersBlock/PodFiltersBlock';
import { PodRow } from '../PodRow/PodRow';
import { usePodStatusView } from '../../hooks';

import classes from './PodList.module.css';

const requestKeyNamespace = 'StageStatus';
const statusRequestKeys = {
   pods: `${requestKeyNamespace}-pods`,
};

interface Props {
   stageId: string;
   duId: string;
   replicaSetId: string;
   /**
    * не то же самое, что `cluster`, для MCRS `'xdc'`
    */
   location: YpLocation;
   cluster: string;
}

/**
 * Список подов c фильтрацией, обновлением и пагинацией.
 */
export const PodList: React.FC<Props> = React.memo(({ stageId, duId, cluster, location, replicaSetId }) => {
   const { search } = useLocation();
   const history = useHistory();
   const urlParams = useMemo(() => new URLSearchParams(search), [search]);

   const [podFilters, setPodFilters] = useState(getFiltersFromUrl(urlParams));
   const isDefaultFilters = useMemo(() => isEqual(podFilters, defaultPodFilters), [podFilters]);

   const [podPagination, setPodPagination] = useState(getPaginationFromUrl(urlParams));

   const [loadingPage, setLoadingPage] = useState(false);

   useEffect(() => {
      // сброс пагинации при смене кластера
      setPodPagination(pagination => ({ ...pagination, page: 1 }));
   }, [cluster]);

   const [activePod, setActivePod] = useState('');
   const [fullMode, setFullMode] = useState(false);

   const { deployUnitStatus } = useDeployUnit(stageId, duId);
   const { replicaSetsInfo, targetRevision } = deployUnitStatus ?? {};
   const currentReplicaSetInfo = replicaSetsInfo?.[createKey({ location, id: replicaSetId })];
   const { type, id } = currentReplicaSetInfo ?? {};
   const { replicaSet } = useReplicaSet({ location, id: id ?? '', type: type ?? ReplicaSetType.PerCluster });

   const { podsSummaryByLocation } = replicaSet ?? {};

   // TODO: заменить на ручку агрегации для точного подсчёта подов
   const podsSummary = podsSummaryByLocation?.[cluster]?.total; // @nodejsgirl: undefined = unknown

   const dispatch = useDispatch();

   const setLoadedPage = useCallback(() => {
      setLoadingPage(false);
   }, []);

   const { activate } = useRequestControl([statusRequestKeys.pods], {
      onSuccess: setLoadedPage,
      onError: setLoadedPage,
   });

   const podSetId = PodConverter.getPodSetId(stageId, duId);
   const podFilter = getFetchQuery(podFilters, targetRevision ?? 0);

   const loadPods = useCallback(() => {
      activate(() => {
         dispatch(
            fetchPods.withRequestKey(statusRequestKeys.pods)({
               podSetId,
               location: cluster as YpLocation,
               limit: podPagination.limit,
               page: podPagination.page,
               filter: podFilter,
            }),
         );
      });
   }, [activate, dispatch, podSetId, cluster, podPagination.limit, podPagination.page, podFilter]);

   useUpdateTimer({
      callback: loadPods,
      fast: TIMES_IN_MS.Second * 10,
      slow: TIMES_IN_MS.Minute,
      mode: UpdateTimerMode.Fast,
   });

   useEffect(() => {
      // первая загрузка
      loadPods();
   }, [loadPods]);

   const podSelector = useCallback(
      (state: RootState) => selectPods(state, { location: cluster, podSetId, filter: podFilter }),
      [cluster, podFilter, podSetId],
   );
   const { pods: rawPods, requestInfo: podsRequestInfo } = useSelector(podSelector);

   const { hasNextPage, hasNoPods } = podsRequestInfo ?? {};

   const podSelectData = rawPods
      .slice(0, podPagination.limit)
      .map(pod => ({ id: pod.meta?.id ?? '', location: cluster }));
   const { podStatusViewList } = usePodStatusView({ podSelectData });

   // явные действия только при вызове функции
   const onChangeFilters = useCallback((filters: PodFilters) => {
      setPodFilters(filters);
   }, []);

   const targetUrlParams = useMemo(() => getUrlParamsFromPodData(podFilters, podPagination), [
      podFilters,
      podPagination,
   ]);
   const targetUrl = urlBuilder.stageStatus(stageId, duId, cluster, targetUrlParams);

   const updateHistory = useCallback(() => history.replace(targetUrl), [history, targetUrl]);
   const updateUrlAction = useSkipAfterFirstRender(
      updateHistory,
      isFirst => !(isFirst && Object.keys(targetUrlParams).length === 0),
   );

   const updateUrl = useRafDelay(updateUrlAction);
   useEffect(() => {
      updateUrl();
   }, [updateUrl]);

   const networkErrors = useNetworkErrors([statusRequestKeys.pods]);
   const podsError = networkErrors[statusRequestKeys.pods];

   const networkRequests = useNetworkRequests([statusRequestKeys.pods]);
   const { state: podRequestState } = networkRequests[statusRequestKeys.pods] ?? {};
   const isLoading = podRequestState === RequestState.PENDING;

   // узнаем лимит на основе интегральных данных
   const isCalculateLimit = isDefaultFilters && !hasNoPods;

   return (
      <div className={classes.podList}>
         {/* @nodejsgirl: undefined = unknown */}
         {podsSummary !== 0 && (
            <>
               <PodFiltersBlock value={podFilters} onChange={onChangeFilters} revision={targetRevision ?? 0} />

               {isLoading && (podStatusViewList.length === 0 || loadingPage) ? (
                  <span data-test={'PodList:Loader'}>
                     <Loader />
                  </span>
               ) : podStatusViewList.length === 0 ? (
                  <div>no pods with selected parameters</div>
               ) : (
                  <table
                     className={classNames(
                        classes.podsTable,
                        classes.largeTable,
                        (podsError?.request?.lastOkTimestamp ?? 0) > 0 ? classes.old : undefined,
                     )}
                  >
                     <thead>
                        <tr>
                           <th colSpan={9}>
                              <div>
                                 <Checkbox
                                    checked={fullMode}
                                    content={'Show additional resources (layers, static resources, volumes)'}
                                    onChange={() => setFullMode(mode => !mode)}
                                 />
                              </div>
                           </th>
                        </tr>
                        <tr>
                           <th aria-label={'Check'} />
                           <th>№</th>
                           <th>FQDN</th>
                           <th>Scheduling</th>
                           <th>Agent</th>
                           <th>Status</th>
                           <th>Revision</th>
                           <th>Host</th>
                           <th>Logs</th>
                        </tr>
                     </thead>
                     <tbody>
                        {podStatusViewList
                           .sort((a1, a2) => (a1.id > a2.id ? 1 : -1))
                           .map((pod, index) => {
                              const { id: podId } = pod;
                              const isActive = activePod === podId;
                              const indexN = (podPagination.page - 1) * podPagination.limit + index + 1;

                              return (
                                 <PodRow
                                    key={podId}
                                    stageId={stageId}
                                    duId={duId}
                                    pod={pod}
                                    active={isActive}
                                    index={indexN}
                                    cluster={cluster}
                                    toggle={() => setActivePod(activeId => (activeId === podId ? '' : podId))}
                                    fullMode={fullMode}
                                 />
                              );
                           })}
                     </tbody>
                  </table>
               )}
               {!(podPagination.page === 1 && hasNoPods) && podsSummary !== undefined && (
                  <div data-test={'pods-filter--pagination'}>
                     <Pagination
                        value={podPagination.page}
                        total={
                           isCalculateLimit
                              ? Math.ceil(podsSummary / podPagination.limit)
                              : podPagination.page + (hasNextPage ? 1 : 0)
                        }
                        maxButtons={isCalculateLimit ? 9 : 1}
                        perPage={podPagination.limit}
                        perPageVariants={podsPerPages}
                        onChange={(e, page) => {
                           setLoadingPage(true);
                           setPodPagination(pagination => ({
                              ...pagination,
                              page,
                           }));
                        }}
                        onPerPageChange={(e, limit) => {
                           setLoadingPage(true);
                           statusSettingsStorage.setItem<number>(podsPaginationPerPageKey, limit);
                           setPodPagination(pagination => ({
                              ...pagination,
                              limit,
                           }));
                        }}
                     />
                  </div>
               )}
            </>
         )}

         {podsError?.error && (
            <div>
               <YpErrorTooltip error={podsError.error} request={podsError.request} />
            </div>
         )}
      </div>
   );
});

PodList.displayName = 'PodList';
