import React, { ReactNode, useCallback, useMemo } from 'react';

import { modalService } from '@yandex-infracloud-ui/libs';

import { StatusState } from '../../../../../../components/lib';
import { PodObjectName, PodObjectTitle, PodObjectType } from '../../../../../../models/ui';
import { RuntimeErrorModal } from '../RuntimeErrorModal/RuntimeErrorModal';
import { RuntimeStatus } from '../RuntimeStatus/RuntimeStatus';
import { RuntimeWarning } from '../RuntimeWarning/RuntimeWarning';
import { RuntimeObjectRevision } from '../RuntimeObjectRevision/RuntimeObjectRevision';

import classes from './PodObjectStatus.module.css';

interface Props<T extends PodObjectName, S extends string | number> {
   type: T;
   podObject: PodObjectType[T];
   state: S;
   statusMap: Record<S, StatusState>;
   inline?: boolean;
}

export const PodObjectStatus = <T extends PodObjectName, S extends string | number>({
   type,
   podObject,
   state,
   statusMap,
   inline = false,
}: Props<T, S>) => {
   const { statusStateInfo, id, warning, warningCount, revision, targetRevision } = podObject;
   const inlineStatus = useMemo(
      () => (
         <RuntimeStatus
            state={state}
            statusMap={statusMap}
            statusStateInfo={statusStateInfo}
            existErrors={warningCount > 0}
            inline={true}
         />
      ),
      [state, statusMap, statusStateInfo, warningCount],
   );

   const openError = useCallback(() => {
      modalService
         .open(RuntimeErrorModal, {
            title: PodObjectTitle[type] as string,
            warningCount,
            id,
            errors: (<RuntimeWarning level={warning} />) as ReactNode,
            additionalObjectData: (
               <div className={classes.objectData}>
                  {inlineStatus}
                  <RuntimeObjectRevision
                     revision={revision}
                     targetRevision={targetRevision}
                     progress={statusStateInfo.inProgress.active}
                     full={true}
                  />
               </div>
            ) as ReactNode,
         })
         .subscribe();
   }, [id, inlineStatus, type, warning, warningCount, statusStateInfo, revision, targetRevision]);

   if (inline) {
      return inlineStatus;
   }

   return (
      <RuntimeStatus
         state={state}
         statusMap={statusMap}
         statusStateInfo={statusStateInfo}
         existErrors={warningCount > 0}
         openError={openError}
      />
   );
};
