import React from 'react';

import { faChevronDown, faChevronRight } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { classNames, ExternalLink } from '@yandex-infracloud-ui/libs';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../../../../models';
import { Pod } from '../../../../../../models/ui';
import { handleStopPropagation } from '../../../../../../utils';
import { PodAgentStatus } from '../PodAgentStatus/PodAgentStatus';
import { PodStatus } from '../PodStatus/PodStatus';
import { SchedulingStatus } from '../SchedulingStatus/SchedulingStatus';
import { PodObjects } from './components/PodObjects/PodObjects';
import { CopyWrapper } from '../../../../../../components/lib';
import { makeLogsQuery } from './utils';
import { RuntimeObjectRevision } from '../RuntimeObjectRevision/RuntimeObjectRevision';

import classes from './PodRow.module.css';

interface Props {
   stageId: string;
   duId: string;
   cluster: string;
   index: number;
   pod: Pod;
   active: boolean;
   toggle(): void;
   fullMode?: boolean;
}

export const PodRow: React.FC<Props> = React.memo(
   ({ pod, stageId, duId, index, active, cluster, toggle, fullMode = false }) => {
      const {
         id: podId,
         nodeId,
         persistentFqdn,
         revision,
         targetRevision,
         statusStateInfo,
         schedulingInfo,
         currentState,
      } = pod;

      return (
         <React.Fragment key={`pod-${cluster}-${podId}`}>
            <tr
               className={classNames(classes.row, { [classes.activeRow]: active })}
               onClick={() => toggle()}
               data-test={'status--pod'}
               data-e2e={`StatusPage:Pods:${index}`}
            >
               <td className={classes.chevron}>
                  <FontAwesomeIcon icon={active ? faChevronDown : faChevronRight} size={'sm'} />
               </td>
               <td data-test={'pod--index'}>{index}</td>
               <td data-test={'pod--fqdn'}>
                  {persistentFqdn && (
                     <CopyWrapper text={persistentFqdn}>
                        {cluster && podId ? (
                           <span data-test={'pod--fqdn'}>
                              <Link to={urlBuilder.ypPod(cluster, podId)} onClick={handleStopPropagation}>
                                 {persistentFqdn}
                              </Link>
                           </span>
                        ) : (
                           <span data-test={'pod--fqdn'}>{persistentFqdn}</span>
                        )}
                     </CopyWrapper>
                  )}
               </td>
               <td>
                  <SchedulingStatus state={schedulingInfo.state} />
               </td>
               <td>
                  <PodAgentStatus state={currentState} />
               </td>
               <td>
                  <PodStatus pod={pod} cluster={cluster} />
               </td>
               <td data-test={'pod--revision'}>
                  <RuntimeObjectRevision
                     revision={revision}
                     targetRevision={targetRevision}
                     progress={statusStateInfo.inProgress.active}
                  />
               </td>
               <td>
                  {cluster && nodeId ? (
                     <span data-test={'pod--host'}>
                        <CopyWrapper text={nodeId}>
                           <Link to={urlBuilder.ypNode(cluster, nodeId)} onClick={handleStopPropagation}>
                              {nodeId}
                           </Link>
                        </CopyWrapper>
                        <div className={classes.links}>
                           <span
                              data-test={'pod--wall-e-link'}
                              role={'link'}
                              tabIndex={2 * index}
                              onClick={handleStopPropagation}
                           >
                              <ExternalLink href={`https://wall-e.yandex-team.ru/host/${nodeId}`}>Wall-E</ExternalLink>
                           </span>
                           <span
                              data-test={'pod--yasm-link'}
                              role={'link'}
                              tabIndex={2 * index + 1}
                              onClick={handleStopPropagation}
                           >
                              <ExternalLink href={`https://yasm.yandex-team.ru/template/panel/Host/hosts=${nodeId}`}>
                                 YASM
                              </ExternalLink>
                           </span>
                        </div>
                     </span>
                  ) : (
                     <span data-test={'pod--host'}>{nodeId}</span>
                  )}
               </td>
               <td>
                  <Link
                     to={urlBuilder.stageLogs(stageId, {
                        deployUnitId: duId,
                        query: makeLogsQuery({
                           node_fqdn: nodeId || undefined,
                           pod: podId,
                        }),
                     })}
                     onClick={handleStopPropagation}
                  >
                     Logs
                  </Link>
               </td>
            </tr>
            {active && (
               <tr data-test={'pod--additional'} className={classes.additional}>
                  <td />
                  <td />
                  <td colSpan={11}>
                     <PodObjects stageId={stageId} duId={duId} pod={pod} cluster={cluster} fullMode={fullMode} />
                  </td>
               </tr>
            )}
         </React.Fragment>
      );
   },
);

PodRow.displayName = 'PodRow';
