import React, { ReactNode } from 'react';

import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
// import { faWrench } from '@fortawesome/pro-regular-svg-icons';
import { faWrench } from '@fortawesome/free-solid-svg-icons';
import { Link, Tooltip, Spin } from '@yandex-cloud/uikit';

import { urlBuilder } from '../../../../../../../../models';
import {
   Pod,
   PodObject,
   PodObjectName,
   PodObjectTitle,
   PodObjectType,
   WarningLevelData,
} from '../../../../../../../../models/ui';
import { handleStopPropagation } from '../../../../../../../../utils';
import { BoxIp } from '../../../BoxIp/BoxIp';
import { makeLogsQuery } from '../../utils';

import classes from './PodObjects.module.css';
import { useStage } from '../../../../../../../../redux';
import { RuntimeObjectRevision } from '../../../RuntimeObjectRevision/RuntimeObjectRevision';
import { PodObjectDataConfig } from '../../../../model';

interface RowProps<T extends PodObjectName> {
   type: T;
   object?: PodObjectType[T];
   targetRevision: number | null;
   logs?: string;
   links?: ReactNode;
   warning?: WarningLevelData<string>;
   isSystem?: boolean;
}

const getPodObjectChecker = <T extends PodObjectName>(XObjectName: T) => (
   type: PodObjectName,
   object: PodObject,
): object is PodObjectType[T] => type === XObjectName;
const isBox = getPodObjectChecker(PodObjectName.Box);
const isLayer = getPodObjectChecker(PodObjectName.Layer);
const isStaticResource = getPodObjectChecker(PodObjectName.StaticResource);

function PodObjectRow<T extends PodObjectName>({ type, object, targetRevision, logs, links, isSystem }: RowProps<T>) {
   if (!object) {
      return null;
   }

   const { id, revision, statusStateInfo } = object;
   const title = PodObjectTitle[type];
   const { icon, isBoxChild, status } = PodObjectDataConfig[type](object as any);
   const downloadProgress = (isLayer(type, object) || isStaticResource(type, object)) && object.downloadProgress;
   const isSystemBox = isBox(type, object) && object.isSystemBox;
   return (
      <tr className={isSystem ? classes.systemObject : undefined}>
         <td>
            {isBoxChild && <div className={classes.workloadLevel} />}
            <FontAwesomeIcon icon={icon} className={isBoxChild ? classes.workloadIcon : classes.boxIcon} />
            <span>{title}</span>
            {isSystemBox && (
               <Tooltip content={'System box'}>
                  <FontAwesomeIcon icon={faWrench} className={classes.systemIcon} />
               </Tooltip>
            )}
         </td>
         <td className={classes.name} title={id}>
            <span className={classes.nameText}>{id}</span>
            {downloadProgress && downloadProgress.percent < 100 && (
               <span className={classes.downloadProgress}>
                  <Spin size={'xs'} /> ...{downloadProgress.percent}%
               </span>
            )}
         </td>
         <td>{status}</td>
         <td>
            <RuntimeObjectRevision
               revision={revision}
               targetRevision={targetRevision}
               progress={statusStateInfo.inProgress.active}
            />
         </td>
         <td>
            {logs && (
               <Link href={logs} onClick={handleStopPropagation}>
                  logs
               </Link>
            )}
         </td>
         <td>{links}</td>
      </tr>
   );
}

interface Props {
   pod: Pod;
   stageId: string;
   duId: string;
   cluster: string;
   fullMode?: boolean;
}

export const PodObjects: React.FC<Props> = React.memo(({ pod, cluster, stageId, duId, fullMode = false }) => {
   const {
      workloadMap,
      layerMap,
      staticResourceMap,
      staticResources,
      volumes,
      layers,
      boxes,
      workloads,
      id: podId,
      persistentFqdn,
      nodeId,
      targetRevision,
   } = pod;

   const { stage } = useStage(stageId);

   if (!stage) {
      return null;
   }

   if (boxes.size === 0) {
      return <span>No boxes for pod {podId}</span>;
   }

   const volumeList = Array.from(volumes.keys())
      .filter(volumeId => fullMode || volumes.get(volumeId)!.warningCount > 0)
      .sort()
      .map(volumeId => (
         <PodObjectRow
            key={`${volumeId}`}
            type={PodObjectName.Volume}
            object={volumes.get(volumeId)}
            targetRevision={targetRevision}
         />
      ));

   const showVolumes = volumeList.length > 0;

   return (
      <table>
         <thead>
            <tr>
               <th>Unit</th>
               <th>Name</th>
               <th>Status</th>
               <th>Revision</th>
               <th>Logs</th>
               <th aria-label={'Box IP'} />
            </tr>
         </thead>
         <tbody>
            {Array.from(boxes.keys())
               .sort((A, B) => {
                  const boxA = boxes.get(A)!;
                  const boxB = boxes.get(B)!;
                  const systemOrder = Number(boxA.isSystemBox) - Number(boxB.isSystemBox);
                  const nameOrder = A > B ? 1 : -1;
                  return systemOrder === 0 ? nameOrder : systemOrder;
               })
               .map(boxId => {
                  const box = boxes.get(boxId)!;
                  const boxWorkloads = Array.from(workloadMap.get(boxId) ?? []);
                  const boxLayers = Array.from(layerMap.get(boxId) ?? []);
                  const boxStaticResources = Array.from(staticResourceMap.get(boxId) ?? []);
                  const isSystem = box.isSystemBox;

                  const layerList = boxLayers
                     .filter(layerId => fullMode || ((layers.get(layerId) ?? {}).warningCount ?? 0) > 0)
                     .sort()
                     .map(layerId => (
                        <PodObjectRow
                           key={`${layerId}`}
                           type={PodObjectName.Layer}
                           object={layers.get(layerId)}
                           targetRevision={targetRevision}
                           isSystem={isSystem}
                        />
                     ));
                  const staticResourceList = boxStaticResources
                     .filter(resourceId => fullMode || ((staticResources.get(resourceId) ?? {}).warningCount ?? 0) > 0)
                     .sort()
                     .map(resourceId => (
                        <PodObjectRow
                           key={`${resourceId}`}
                           type={PodObjectName.StaticResource}
                           object={staticResources.get(resourceId)}
                           targetRevision={targetRevision}
                           isSystem={isSystem}
                        />
                     ));
                  const showAdditional = layerList.length + staticResourceList.length > 0;
                  return (
                     <React.Fragment key={`box-${boxId}-${podId}-${cluster}`}>
                        <PodObjectRow
                           key={`${boxId}`}
                           type={PodObjectName.Box}
                           object={box}
                           targetRevision={targetRevision}
                           logs={urlBuilder.stageLogs(stageId, {
                              deployUnitId: duId,
                              query: makeLogsQuery({
                                 node_fqdn: nodeId || undefined,
                                 pod: podId,
                                 box: boxId,
                              }),
                           })}
                           links={persistentFqdn && <BoxIp box={box} fqdn={persistentFqdn} isSox={stage.soxService} />}
                           isSystem={isSystem}
                        />
                        {boxWorkloads.sort().map(workloadId => (
                           <PodObjectRow
                              key={`${workloadId}`}
                              type={PodObjectName.Workload}
                              object={workloads.get(workloadId)}
                              targetRevision={targetRevision}
                              logs={urlBuilder.stageLogs(stageId, {
                                 deployUnitId: duId,
                                 query: makeLogsQuery({
                                    node_fqdn: nodeId || undefined,
                                    pod: podId,
                                    box: boxId,
                                    workload: workloadId,
                                 }),
                              })}
                              isSystem={isSystem}
                           />
                        ))}
                        {showAdditional && (
                           <>
                              <tr>
                                 <td colSpan={6}>
                                    <div className={classes.boxHeadBlock}>
                                       <div className={classes.resourcesHead} />
                                       <span className={classes.objectType}>Layers, Static Resources</span>
                                    </div>
                                 </td>
                              </tr>
                              {layerList}
                              {staticResourceList}
                           </>
                        )}
                     </React.Fragment>
                  );
               })}
            {showVolumes && (
               <>
                  <tr>
                     <td colSpan={6}>
                        <div className={classes.resourcesHead} />
                        <span className={classes.objectType}>Volumes</span>
                     </td>
                  </tr>
                  {volumeList}
               </>
            )}
         </tbody>
      </table>
   );
});

PodObjects.displayName = 'PodObjects';
