import React, { ReactNode, useCallback, useMemo } from 'react';

import { formatDate, modalService } from '@yandex-infracloud-ui/libs';

import { Pod, PodCustomCurrentState } from '../../../../../../models/ui';
import { RuntimeErrorModal } from '../RuntimeErrorModal/RuntimeErrorModal';
import { SimpleModal } from '../../../../../../components/lib';
import { statusMap } from '../../model';
import { PodError } from '../PodError/PodError';
import { RuntimeStatus } from '../RuntimeStatus/RuntimeStatus';
import { RuntimeObjectRevision } from '../RuntimeObjectRevision/RuntimeObjectRevision';

import classes from './PodStatus.module.css';

interface Props {
   pod: Pod;
   cluster?: string;
   inline?: boolean;
}

export const PodStatus: React.FC<Props> = React.memo(({ pod, cluster, inline }) => {
   const { statusStateInfo, id: podId, currentState, totalWarningCount, agentLastHeartbeatTime } = pod;

   const inlineStatus = useMemo(
      () => (
         <RuntimeStatus
            state={currentState}
            statusMap={statusMap}
            statusStateInfo={statusStateInfo}
            existErrors={totalWarningCount > 0}
            inline={true}
         />
      ),
      [currentState, statusStateInfo, totalWarningCount],
   );

   const openSuspectMessage = useCallback(() => {
      modalService
         .open(SimpleModal, {
            title: 'Pod message',
            children: (
               <div className={classes.podData}>
                  {agentLastHeartbeatTime &&
                     `The last pod heartbeat was on ${formatDate(agentLastHeartbeatTime / 1000)}. `}
                  This may mean that the pod host is dead.
               </div>
            ),
         })
         .subscribe();
   }, [agentLastHeartbeatTime]);

   const openError = useCallback(() => {
      modalService
         .open(RuntimeErrorModal, {
            title: 'Pod',
            warningCount: totalWarningCount,
            id: podId,
            location: cluster,
            errors: (<PodError pod={pod} />) as ReactNode,
            additionalObjectData: (
               <div className={classes.podData}>
                  {inlineStatus}
                  <RuntimeObjectRevision
                     revision={pod.revision}
                     targetRevision={pod.targetRevision}
                     progress={statusStateInfo.inProgress.active}
                     full={true}
                  />
               </div>
            ) as ReactNode,
         })
         .subscribe();
   }, [cluster, totalWarningCount, podId, pod, statusStateInfo, inlineStatus]);

   if (inline) {
      return inlineStatus;
   }

   return (
      <RuntimeStatus
         state={currentState}
         statusMap={statusMap}
         statusStateInfo={statusStateInfo}
         existMessage={currentState === PodCustomCurrentState.PCS_SUSPECTED}
         openMessage={openSuspectMessage}
         openError={openError}
         existErrors={totalWarningCount > 0}
      />
   );
});

PodStatus.displayName = 'PodStatus';
