import { faLongArrowRight } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Popover, Spin } from '@yandex-cloud/uikit';
import { classNames, formatDate, Json, TIMES_IN_MS, UpdateTimerMode, useUpdateTimer } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import { PerClusterApprove } from '../../../../../../components';
import { YpErrorTooltip } from '../../../../../../components/network';
import { isNonExistenceYpError } from '../../../../../../models/api';
import { PodConverter, ReplicaSetInfo, ReplicaSetRequestKeys, ReplicaSetType } from '../../../../../../models/ui';
import { getReplicaSetLocationInfo } from '../../../../../../models/ui/status';
import { EObjectType } from '../../../../../../proto-typings';
import { getReplicaSet, RootState, selectDuPods, useNetworkErrors } from '../../../../../../redux';
import { useConfig } from '../../../../../../services';
import { useDeployUnitStatusView, usePodStatusView, useReplicaSetStatusView, useStageStatusView } from '../../hooks';
import { statusSettingsStorage, statusViewSettingsVisible } from '../../model';

import { MergePodInfo } from './components/MergePodInfo/MergePodInfo';
import { PodsSummaryLine } from './components/PodsSummaryLine/PodsSummaryLine';
import { ReplicaSetPending } from './components/ReplicaSetPending/ReplicaSetPending';
import { ReplicaSetRevision } from './components/ReplicaSetRevision/ReplicaSetRevision';
import { ReplicaSetStatus } from './components/ReplicaSetStatus/ReplicaSetStatus';
import classes from './ReplicaSetTile.module.css';

interface Props {
   replicaSetInfo: ReplicaSetInfo;
   stageId: string;
   duId: string;
   activeClusters: Set<string>;
}

export const ReplicaSetTile: React.FC<Props> = React.memo(({ replicaSetInfo, stageId, duId, activeClusters }) => {
   const { id, locations, type, endpointSets } = replicaSetInfo;

   const { replicaSetLocationInfo } = getReplicaSetLocationInfo({ replicaSetData: replicaSetInfo });
   const { oneLocation, ypLocation } = replicaSetLocationInfo;

   const requestKey = ReplicaSetRequestKeys.getOne({
      id,
      location: ypLocation,
   });

   const dispatch = useDispatch();

   const refreshReplicaSetData = useCallback(() => {
      dispatch(
         getReplicaSet.withRequestKey(requestKey)({
            id,
            location: ypLocation,
            type:
               type === ReplicaSetType.MultiCluster
                  ? EObjectType.OT_MULTI_CLUSTER_REPLICA_SET
                  : EObjectType.OT_REPLICA_SET,
         }),
      );
   }, [dispatch, id, requestKey, type, ypLocation]);

   useEffect(() => {
      refreshReplicaSetData();
   }, [refreshReplicaSetData]);

   useUpdateTimer({
      callback: refreshReplicaSetData,
      fast: 10 * TIMES_IN_MS.Second,
      slow: TIMES_IN_MS.Minute,
      mode: UpdateTimerMode.Fast,
   });

   const { stageStatusViewList } = useStageStatusView({ stageSelectData: [{ stageId }] });
   const [stageStatusView = null] = stageStatusViewList;

   const replicaSetSelectItem = {
      id,
      location: ypLocation,
   };
   const { replicaSetStatusViewList } = useReplicaSetStatusView({
      replicaSetSelectData: [replicaSetSelectItem],
   });
   const [replicaSetStatusView = null] = replicaSetStatusViewList;

   const { deployUnitStatusViewList } = useDeployUnitStatusView({ deployUnitSelectData: [{ stageId, duId }] });
   const [deployUnitStatusView = null] = deployUnitStatusViewList;

   const networkErrors = useNetworkErrors([requestKey]);
   const replicaSetError = networkErrors[requestKey];

   const { clusters } = useConfig()!;

   const podsSelector = useCallback(
      (state: RootState) => selectDuPods(state, PodConverter.getPodSetId(stageId, duId)),
      [duId, stageId],
   );

   const podStore = useSelector(podsSelector);
   const podSelectData = useMemo(
      () =>
         Object.keys(podStore)
            .filter(cluster => locations?.has(cluster))
            .flatMap(cluster =>
               podStore?.[cluster].map(pod => ({
                  location: cluster,
                  id: pod.meta?.id ?? '',
               })),
            ),
      [locations, podStore],
   );

   const { podStatusViewList } = usePodStatusView({ podSelectData });

   const replicaSetExists = !isNonExistenceYpError(replicaSetError?.error?.error ?? null);

   if (!deployUnitStatusView) {
      return null;
   }

   const { currentRevision, targetRevision, labels, replicaSetDisruptionBudgets } = replicaSetStatusView ?? {};

   const isClusterIgnored =
      type === ReplicaSetType.PerCluster && stageStatusView?.disabledLocations.has(oneLocation ?? '');

   const locationsTitles = clusters
      .map(e => e.value)
      .filter(e => locations.has(e))
      .map((e, i) => (
         <>
            {i > 0 && ', '}
            <span style={stageStatusView?.disabledLocations.has(e) ? { color: 'var(--yc-color-text-secondary)' } : {}}>
               {e.toUpperCase()}
            </span>
         </>
      ));

   const { locationsWaitingForApprove } = deployUnitStatusView;

   const { override: overrideDisruptionBudget, spec: specDisruptionBudget } = replicaSetDisruptionBudgets ?? {};

   const disruptionBudgetChanged =
      overrideDisruptionBudget && overrideDisruptionBudget.value !== (specDisruptionBudget?.value ?? 0);

   const waitingForApprove =
      type === ReplicaSetType.PerCluster ? locationsWaitingForApprove.get(oneLocation ?? '') ?? false : false;

   const viewPodMerge = statusSettingsStorage.getItem<boolean>(statusViewSettingsVisible, false);

   return (
      <div
         className={classNames(classes.tile, {
            [classes.oldData]: replicaSetError?.request?.lastOkTimestamp,
            [classes.ignoreRS]: isClusterIgnored,
         })}
         data-test={'status--replica-set'}
      >
         <>
            <div className={classes.header}>
               <h3 data-test={'replica-set--cluster'}>{locationsTitles}</h3>
               <div className={classes.revision}>
                  <ReplicaSetRevision
                     currentRevision={currentRevision ?? null}
                     targetRevision={targetRevision ?? null}
                     inProgress={replicaSetStatusView?.statusStateInfo?.inProgress?.active}
                  />
               </div>
               <div>
                  {waitingForApprove ? (
                     <PerClusterApprove stageId={stageId} duId={duId} cluster={oneLocation!} />
                  ) : (
                     replicaSetStatusView && (
                        <ReplicaSetStatus replicaSet={replicaSetStatusView} pods={podStatusViewList} />
                     )
                  )}
               </div>
            </div>
            {replicaSetStatusView ? (
               <div className={classes.content}>
                  <div className={classes.pods}>
                     <div>
                        <b data-test={'replica-set--pods-total'}>{replicaSetStatusView.podsSummary.total}</b> Pods total
                        {viewPodMerge && <MergePodInfo replicaSetStatusView={replicaSetStatusView} />}
                     </div>
                     <div className={classes.podsLine}>
                        <PodsSummaryLine podsSummary={replicaSetStatusView.mergePodsSummary} />
                     </div>
                  </div>
                  <div className={classes.info}>
                     <div>
                        <div>Endpoints: {Object.values(endpointSets).reduce((sum, e) => sum + e.size, 0)}</div>
                        <div>
                           Budget:{' '}
                           <span className={classNames(disruptionBudgetChanged ? classes.override : undefined)}>
                              {specDisruptionBudget?.value ?? 0}
                           </span>
                           {disruptionBudgetChanged && (
                              <span>
                                 {' '}
                                 <FontAwesomeIcon icon={faLongArrowRight} /> {overrideDisruptionBudget!.value}
                              </span>
                           )}
                        </div>
                     </div>
                     {labels && Object.keys(labels).length > 0 && Object.values(labels).every(Boolean) && (
                        <div className={classes.labels}>
                           <Popover content={<Json obj={labels} hidePre />}>Labels</Popover>
                        </div>
                     )}
                  </div>
               </div>
            ) : (
               replicaSetExists && (
                  <div className={classes.loading}>
                     <Spin size={'s'} />
                  </div>
               )
            )}
            {replicaSetError?.error &&
               (replicaSetExists ? (
                  <div className={classes.error}>
                     <YpErrorTooltip error={replicaSetError.error} request={replicaSetError.request} />
                     {replicaSetError.request?.lastOkTimestamp && (
                        <div className={classes.lastUpdate}>
                           The last update was at{' '}
                           {formatDate(new Date(replicaSetError.request.lastOkTimestamp), 'HH:mm:ss')}
                        </div>
                     )}
                  </div>
               ) : (
                  <ReplicaSetPending error={replicaSetError.error} request={replicaSetError.request} />
               ))}
            {isClusterIgnored && (
               <div className={classes.disabledCluster}>
                  <h3 data-test={'replica-set--cluster'}>{locationsTitles} is shut off permanently.</h3>
               </div>
            )}
         </>
      </div>
   );
});

ReplicaSetTile.displayName = 'ReplicaSetTile';
