import React, { ReactNode, useCallback, useMemo } from 'react';

import { modalService } from '@yandex-infracloud-ui/libs';

import { getReplicaSetWarning } from '../../../../../../../../models/ui';
import { ReplicaSetError } from '../../../ReplicaSetError/ReplicaSetError';
import { RuntimeErrorModal } from '../../../RuntimeErrorModal/RuntimeErrorModal';
import { RuntimeStatus } from '../../../RuntimeStatus/RuntimeStatus';
import { PodError } from '../../../PodError/PodError';
import { YpObjectErrorBlock } from '../../../DeployUnitErrors/DeployUnitErrors';
import { PodStatus } from '../../../PodStatus/PodStatus';
import { RuntimeObjectRevision } from '../../../RuntimeObjectRevision/RuntimeObjectRevision';

import classes from './ReplicaSetStatus.module.css';
import { PodStatusView, ReplicaSetStatusView } from '../../../../../../../../models/ui/status';
import { statusSettingsStorage, statusViewSettingsVisible } from '../../../../model';
import { MergeStatusInfo } from '../MergeStatusInfo/MergeStatusInfo';

interface Props {
   replicaSet: ReplicaSetStatusView;
   pods: PodStatusView[];
   inline?: boolean;
}

export const ReplicaSetStatus: React.FC<Props> = React.memo(({ replicaSet, pods, inline }) => {
   const { id, location, mergeStatusStateInfo } = replicaSet;

   const replicaSetWarning = getReplicaSetWarning(replicaSet);
   const podWarningCount = pods.reduce((sum, pod) => sum + pod.totalWarningCount, 0);
   const warningCount = replicaSetWarning.count + podWarningCount;
   const existErrors = warningCount > 0;

   const inlineStatus = useMemo(
      () => (
         <RuntimeStatus
            state={null}
            statusMap={{}}
            statusStateInfo={mergeStatusStateInfo}
            existErrors={existErrors}
            inline={true}
         />
      ),
      [existErrors, mergeStatusStateInfo],
   );

   const openError = useCallback(() => {
      modalService
         .open(RuntimeErrorModal, {
            title: 'Replica set',
            warningCount,
            id,
            location: location as string | undefined,
            errors: (
               <div className={classes.errorList}>
                  <ReplicaSetError warning={replicaSetWarning} />
                  {pods.map((pod, i) => (
                     <YpObjectErrorBlock
                        key={pod.id}
                        title={'Pod'}
                        id={pod.id}
                        count={pod.totalWarningCount}
                        location={pod.location}
                        additionalData={
                           <div className={classes.objectData}>
                              <PodStatus pod={pod} inline={true} />
                              <RuntimeObjectRevision
                                 revision={pod.revision}
                                 targetRevision={pod.targetRevision}
                                 progress={pod.statusStateInfo.inProgress.active}
                                 full={true}
                              />
                           </div>
                        }
                        expand={i === 0}
                     >
                        <PodError pod={pod} />
                     </YpObjectErrorBlock>
                  ))}
               </div>
            ) as ReactNode,
            additionalObjectData: inlineStatus as ReactNode,
         })
         .subscribe();
   }, [id, location, pods, replicaSetWarning, warningCount, inlineStatus]);

   const viewPodMerge = statusSettingsStorage.getItem<boolean>(statusViewSettingsVisible, false);

   let content: ReactNode = inlineStatus;
   if (!inline) {
      content = (
         <RuntimeStatus
            state={null}
            statusMap={{}}
            statusStateInfo={mergeStatusStateInfo}
            existErrors={existErrors}
            openError={openError}
         />
      );
   }

   return (
      <div className={classes.statusContainer}>
         {content}
         {viewPodMerge && <MergeStatusInfo replicaSetStatusView={replicaSet} existErrors={existErrors} />}
      </div>
   );
});

ReplicaSetStatus.displayName = 'ReplicaSetStatus';
