import React, { useCallback, useContext, useEffect, useMemo, useState } from 'react';

import { ErrorBoundary } from '@yandex-infracloud-ui/libs';
import { useHistory } from 'react-router';

import { TilesTabs } from '../../../../../../components/lib';
import { urlBuilder } from '../../../../../../models';
import { useDeployUnit } from '../../../../../../redux/hooks/useDeployUnit';
import { useConfig } from '../../../../../../services';
import { restoreObjectFromKey } from '../../../../../../utils';
import { ClusterContext } from '../../context';
import { ReplicaSetTile } from '../ReplicaSetTile/ReplicaSetTile';
import { ReplicaSetTabs } from './components/ReplicaSetTabs/ReplicaSetTabs';

import classes from './ReplicaSets.module.css';

interface Props {
   stageId: string;
   duId: string;
}

export const ReplicaSets: React.FC<Props> = React.memo(({ stageId, duId }) => {
   const { deployUnitStatus } = useDeployUnit(stageId, duId);
   const { replicaSetsInfo } = deployUnitStatus ?? {};
   const activeCluster = useContext(ClusterContext);
   const activeClusters = useMemo(() => new Set(activeCluster ? [activeCluster] : []), [activeCluster]);

   const { clusterOrder } = useConfig()!;

   const locationsData = useMemo(
      () => Object.keys(replicaSetsInfo ?? {}).map(key => [key, restoreObjectFromKey(key)] as const),
      [replicaSetsInfo],
   );

   const replicaSetList = useMemo(
      () =>
         locationsData.sort(
            ([, aData], [, bData]) =>
               (clusterOrder[aData.location] ?? Infinity) - (clusterOrder[bData.location] ?? Infinity),
         ),
      [clusterOrder, locationsData],
   );

   const findIndex = replicaSetList.map(e => e[1].location).findIndex(e => e === activeCluster);
   const activeIndex = findIndex === -1 ? 0 : findIndex;
   const [selectedIndex, setSelectedIndex] = useState<number | null>(activeIndex);

   useEffect(() => {
      if (activeIndex !== selectedIndex) {
         setSelectedIndex(activeIndex);
      }
   }, [activeIndex, selectedIndex]);

   const history = useHistory();

   const onToogleIndex = useCallback(
      (index: number | null) => {
         if (index === null) {
            history.replace(urlBuilder.stageStatus(stageId, duId));
         } else {
            const data = replicaSetList[index];
            const { location } = data[1];
            if (clusterOrder.hasOwnProperty(location)) {
               // для Per cluster replica set
               history.replace(urlBuilder.stageStatus(stageId, duId, location));
            }
         }
      },
      [clusterOrder, duId, history, replicaSetList, stageId],
   );

   if (!replicaSetsInfo) {
      return null;
   }

   const replicaSetCount = Object.keys(replicaSetsInfo).length;

   if (replicaSetCount === 0) {
      return null;
   }

   return (
      <div className={classes.card}>
         <ErrorBoundary>
            <TilesTabs
               inline={true}
               showCollapse={false}
               tileContentWidth={400}
               selectedTab={selectedIndex}
               onToogle={onToogleIndex}
               items={replicaSetList.map(([key, { location }]) => ({
                  tile: (
                     <ReplicaSetTile
                        replicaSetInfo={replicaSetsInfo[key]}
                        stageId={stageId}
                        duId={duId}
                        activeClusters={activeClusters}
                     />
                  ),
                  content: <ReplicaSetTabs replicaSetInfo={replicaSetsInfo[key]} stageId={stageId} duId={duId} />,
               }))}
            />
         </ErrorBoundary>
      </div>
   );
});

ReplicaSets.displayName = 'ReplicaSets';
