import { Button } from '@yandex-cloud/uikit';
import { StaffCard } from '@yandex-data-ui/common';
import {
   formatNumber,
   InputField2,
   isEqual,
   modalService,
   UserName,
   UserOrGroupType,
   WarningPlate,
} from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikProps } from 'formik';
import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react';
import { number, object } from 'yup';

import { ReplicaSetDisruptionBudgets, ReplicaSetInfo } from '../../../../../../../../models/ui';
import { DeployUnitStatusView } from '../../../../../../../../models/ui/status';
import { noop } from '../../../../../../../../utils';

import classes from './ReplicaSetDisruptionBudget.module.css';
import { ReplicaSetDisruptionBudgetModal } from './ReplicaSetDisruptionBudgetModal';

interface Props {
   replicaSetInfo: ReplicaSetInfo;
   deployUnit: DeployUnitStatusView;
   stageId: string;
   podsTotal: number | undefined;
   disruptionBudgets: ReplicaSetDisruptionBudgets | null;
}

interface FormValues {
   disruptionBudget: number;
}

const validationSchema = object<FormValues>({
   disruptionBudget: number().required().label('Disruption budget').integer().min(0),
});

export const ReplicaSetDisruptionBudget: React.FC<Props> = React.memo(
   ({ replicaSetInfo, deployUnit, stageId, podsTotal, disruptionBudgets }) => {
      const [resetForm, setResetForm] = useState({
         reset: () => {
            noop();
         },
      });

      const { spec: specDisruptionBudget, override: overrideDisruptionBudget } = disruptionBudgets ?? {};

      const disruptionBudgetChanged =
         overrideDisruptionBudget && overrideDisruptionBudget.value !== (specDisruptionBudget?.value ?? 0);

      const overrideDisruptionBudgetValue = overrideDisruptionBudget?.value;

      const currentDisruptionBudgetValue = disruptionBudgetChanged
         ? overrideDisruptionBudgetValue!
         : specDisruptionBudget?.value ?? 0;

      const initialValues: FormValues = useMemo(
         () => ({
            disruptionBudget: currentDisruptionBudgetValue,
         }),
         [currentDisruptionBudgetValue],
      );

      const onSubmit = useCallback(
         (values: typeof initialValues) => {
            modalService
               .open(ReplicaSetDisruptionBudgetModal, {
                  stageId,
                  duId: deployUnit.id,
                  duRevision: deployUnit.spec.revision ?? 0,
                  value: values.disruptionBudget,
                  oldValue: currentDisruptionBudgetValue,
                  clusters: [...replicaSetInfo.locations.values()],
                  replicaSetType: replicaSetInfo.type,
               })
               .subscribe(() => {
                  resetForm.reset();
               });
         },
         [
            stageId,
            deployUnit.id,
            deployUnit.spec.revision,
            currentDisruptionBudgetValue,
            replicaSetInfo.locations,
            replicaSetInfo.type,
            resetForm,
         ],
      );

      const formRef = useRef<FormikProps<FormValues> | null>(null);

      const locationsKey = [...replicaSetInfo.locations.values()].join();

      useEffect(() => {
         if (formRef.current) {
            formRef.current.resetForm({ values: initialValues });
         }
      }, [replicaSetInfo.id, locationsKey, initialValues]);

      return (
         <div>
            <p>Spec value: {specDisruptionBudget?.value ?? 0}</p>
            {overrideDisruptionBudget && (
               <div className={classes.overrideValue}>
                  Overridden value: {overrideDisruptionBudget.value}{' '}
                  {overrideDisruptionBudget.author && (
                     <StaffCard login={overrideDisruptionBudget.author} className={classes.overrideAuthor}>
                        <UserName value={{ id: overrideDisruptionBudget.author, type: UserOrGroupType.People }} />
                     </StaffCard>
                  )}
               </div>
            )}
            <Formik
               initialValues={initialValues}
               onSubmit={values => onSubmit(values)}
               validationSchema={validationSchema}
               validateOnChange
               innerRef={formRef}
            >
               {form => (
                  <Form>
                     <div className={classes.formLayout}>
                        <InputField2
                           name={'disruptionBudget'}
                           label={'Override disruption budget'}
                           controlProps={{
                              controlProps: { min: 0 },
                              type: 'number',
                              className: classes.disruptionBudget,
                           }}
                           hideErrors
                        />
                        <Button
                           onClick={() => {
                              onSubmit(form.values);
                              setResetForm({ reset: () => form.resetForm({ values: initialValues }) });
                           }}
                           disabled={isEqual(form.values, form.initialValues) || !form.isValid}
                           view={'action'}
                        >
                           Update
                        </Button>
                     </div>
                     {form.errors.disruptionBudget && (
                        <div className={classes.error}>{form.errors.disruptionBudget}</div>
                     )}

                     {podsTotal !== undefined && form.values.disruptionBudget > podsTotal && (
                        <WarningPlate>
                           The proposed disruption budget ({formatNumber(form.values.disruptionBudget)}) is larger than
                           pods total count ({formatNumber(podsTotal)})
                        </WarningPlate>
                     )}
                  </Form>
               )}
            </Formik>
         </div>
      );
   },
);

ReplicaSetDisruptionBudget.displayName = 'ReplicaSetDisruptionBudget';
