import React, { useCallback, useMemo, useState } from 'react';

import { Button } from '@yandex-cloud/uikit';
import { DISMISS_REASON, IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikHelpers } from 'formik';
import { object, string } from 'yup';
import { useDispatch } from 'react-redux';

import { OvverideStageDisruptionBudgetOptions } from '../../../../../../../../services/api/services/YpApi';
import { getStages, useNetworkErrors } from '../../../../../../../../redux';
import { ActionState, useRequestControl } from '../../../../../../../../redux/hooks/useRequestControl';
import { YpErrorTooltip } from '../../../../../../../../components/network';
import { overrideStageDisruptionBudget } from '../../../../../../../../redux/slices/yp/slices/stages';
import { ReplicaSetType } from '../../../../../../../../models/ui';

import modalClasses from '../../../../../../../../design/commonModals.module.css';
import classes from './ReplicaSetDisruptionBudgetModal.module.css';

const initialValue = {
   message: '',
};

export type OverrideStageDisruptionBudgetParams = typeof initialValue;

const ticketActionParamsValidationSchema = object<OverrideStageDisruptionBudgetParams>({
   message: string(),
});

type Props = IModalProps<OverrideStageDisruptionBudgetParams> &
   OvverideStageDisruptionBudgetOptions & {
      replicaSetType: ReplicaSetType;
      oldValue: number;
   };

const overrideStageDisruptionBudgetKey = 'overrideStageDisruptionBudget';
const requestKeys = [overrideStageDisruptionBudgetKey];

export const ReplicaSetDisruptionBudgetModal: React.FC<Props> = React.memo(
   ({ stageId, duId, duRevision, clusters, value, oldValue, replicaSetType, ok, cancel }) => {
      const [formikHelpers, setFormikHelpers] = useState<FormikHelpers<OverrideStageDisruptionBudgetParams> | null>(
         null,
      );

      const dispatch = useDispatch();

      const networkErrors = useNetworkErrors(requestKeys);
      const actionError = networkErrors[overrideStageDisruptionBudgetKey];

      const override = useMemo(
         () =>
            overrideStageDisruptionBudget.withRequestKey(overrideStageDisruptionBudgetKey)({
               stageId,
               duId,
               duRevision,
               clusters: replicaSetType === ReplicaSetType.PerCluster ? clusters : ['multi'],
               value,
            }),
         [stageId, duId, duRevision, replicaSetType, clusters, value],
      );

      const breakSubmitting = useCallback(() => {
         if (formikHelpers) {
            formikHelpers.setSubmitting(false);
         }
      }, [formikHelpers]);

      const { activate: overrideActivate, actionStates } = useRequestControl([overrideStageDisruptionBudgetKey], {
         onSuccess: () => {
            dispatch(getStages({ objectIds: [stageId], paths: stage => [stage.status.runtime_deploy_controls] }));
            ok({ message: '' });
            breakSubmitting();
         },
         onError: () => {
            breakSubmitting();
         },
      });

      const handleSubmit = useCallback(
         (v: OverrideStageDisruptionBudgetParams, helpers: FormikHelpers<OverrideStageDisruptionBudgetParams>) => {
            overrideActivate(() => dispatch(override));
            setFormikHelpers(helpers);
         },
         [dispatch, override, overrideActivate],
      );

      const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

      const actionState = actionStates.get(overrideStageDisruptionBudgetKey);
      const pending = actionState === ActionState.Pending;

      const clusterNames = clusters.map(cluster => cluster.toUpperCase());

      return (
         <ModalLayout
            title={`Override disruption budget for ${clusterNames.join(', ')}`}
            showFooter={false}
            onDismiss={dismiss}
            data-e2e={'Stage:OverrideDisruptionBudgetModal'}
         >
            <p className={classes.note}>
               Please confirm that revision <b>{duRevision}</b> of <b>{duId}</b> for <b>{clusterNames.join(', ')}</b>{' '}
               will be deployed with overridden disruption budget ({oldValue} was changed to {value}). Next revision
               will be deployed as usual.
            </p>

            <Formik
               initialValues={initialValue}
               onSubmit={handleSubmit}
               validationSchema={ticketActionParamsValidationSchema}
            >
               {form => (
                  <Form className={modalClasses.form}>
                     <footer className={modalClasses.footer}>
                        <Button view={'normal'} className={modalClasses.button} onClick={dismiss} disabled={pending}>
                           Cancel
                        </Button>

                        <Button
                           view={'action'}
                           className={modalClasses.button}
                           type={'submit'}
                           disabled={!form.isValid || pending}
                        >
                           Update
                        </Button>
                     </footer>
                     {actionState === ActionState.Error && actionError?.error && (
                        <YpErrorTooltip error={actionError.error} request={actionError.request} />
                     )}
                  </Form>
               )}
            </Formik>
         </ModalLayout>
      );
   },
);

ReplicaSetDisruptionBudgetModal.displayName = 'ReplicaSetDisruptionBudgetModal';
