import React, { useCallback, useContext } from 'react';
import { useHistory } from 'react-router';
import { LostClusterWarningPlate } from '../../../../../../../../components/common';
import { urlBuilder } from '../../../../../../../../models';

import { YpLocation } from '../../../../../../../../models/api';
import { ReplicaSetInfo, ReplicaSetType } from '../../../../../../../../models/ui';
import RadioButton from '../../../../../../../../old-code/components/RadioButton/RadioButton';
import { useReplicaSet, useStage } from '../../../../../../../../redux';
import { useConfig } from '../../../../../../../../services';
import { ClusterContext } from '../../../../context';
import { PodList } from '../../../PodList/PodList';

import classes from './ReplicaSetPods.module.css';

interface Props {
   stageId: string;
   duId: string;
   replicaSetInfo: ReplicaSetInfo;
}

export const ReplicaSetPods: React.FC<Props> = React.memo(({ stageId, duId, replicaSetInfo }) => {
   const { id, type, locations } = replicaSetInfo;
   const isMultiCluster = type === ReplicaSetType.MultiCluster;
   const { clusters } = useConfig()!;
   const activeCluster = useContext(ClusterContext);

   const { stage } = useStage(stageId);

   const isDisabled = isMultiCluster && stage?.labels?.deploy?.disabled_clusters?.includes(activeCluster!);

   const locationsList = clusters.map(data => data.value).filter(value => locations.has(value));
   const location: YpLocation = isMultiCluster ? YpLocation.XDC : (activeCluster as YpLocation);

   const { replicaSet } = useReplicaSet({ location, id, type });
   const { podsSummaryByLocation } = replicaSet ?? {};

   const history = useHistory();
   const onChangeCluster = useCallback(
      (cluster: string) => {
         history.replace(urlBuilder.stageStatus(stageId, duId, cluster));
      },
      [duId, history, stageId],
   );

   return (
      <div>
         {isMultiCluster && locations.size > 0 && (
            <div className={classes.locations} data-test={'pods-filter--location'}>
               <RadioButton
                  value={activeCluster}
                  items={locationsList.map(clusterName => ({
                     key: `key--cluster-${clusterName}`,
                     value: clusterName,
                     text: clusterName.toUpperCase(),
                     // рисуем '?' для недоступных кластеров RS https://st.yandex-team.ru/DEPLOY-4823#610d17829fd7920b79cdb1af
                     // для MCRS количество подов мы знаем
                     meta: `${podsSummaryByLocation?.[clusterName]?.total ?? '?'}`,
                  }))}
                  onChange={(e: any) => {
                     // TODO: заменить на новые кнопки
                     onChangeCluster(e.target.value);
                  }}
               />
            </div>
         )}
         {isDisabled && <LostClusterWarningPlate cluster={activeCluster ?? locationsList[0]} isDisabled={isDisabled} />}
         {<PodList stageId={stageId} duId={duId} replicaSetId={id} cluster={activeCluster ?? ''} location={location} />}
      </div>
   );
});

ReplicaSetPods.displayName = 'ReplicaSetPods';
