import React from 'react';
import { ReplicaSetPodsSummary } from '../../../../../../../../models/ui';

import classes from './ReplicaSetStatuses.module.css';

interface Props {
   podsSummary: ReplicaSetPodsSummary;
}

const colors = {
   ready: 'var(--yc-color-infographics-positive-heavy)',
   inProgress: 'var(--yc-color-infographics-info-heavy)',
   failed: 'var(--yc-color-infographics-danger-heavy)',
   oldReady: 'var(--yc-color-infographics-positive-medium)',
   oldInProgress: 'var(--yc-color-infographics-info-medium)',
   oldFailed: 'var(--yc-color-infographics-danger-medium)',
   unavailable: 'var(--yc-color-shadow)',
   unknown: 'var(--yc-color-shadow)',
};

const StatusCeil = ({
   name,
   value,
   title,
}: {
   name: Exclude<keyof ReplicaSetPodsSummary, 'total'>;
   value: number;
   title: string;
}) =>
   value > 0 ? (
      <div className={classes.ceil}>
         <div className={classes.badge} style={{ backgroundColor: colors[name] }} />
         <div>{title}</div>
         <div>{value}</div>
      </div>
   ) : null;

export const ReplicaSetStatuses: React.FC<Props> = React.memo(({ podsSummary }) => {
   const { total, ready, inProgress, failed, unavailable, unknown, oldReady, oldInProgress, oldFailed } = podsSummary;
   const currentCount = ready + inProgress + failed;
   const oldCount = oldReady + oldInProgress + oldFailed;
   const unknownCount = unavailable + unknown;

   if (total === 0) {
      return <span>No pods</span>;
   }

   return (
      <div className={classes.statuses}>
         {currentCount > 0 && (
            <div>
               <div className={classes.title}>Current revision</div>
               <StatusCeil name={'ready'} value={ready} title={'Ready'} />
               <StatusCeil name={'inProgress'} value={inProgress} title={'In progress'} />
               <StatusCeil name={'failed'} value={failed} title={'Failed'} />
            </div>
         )}
         {oldCount > 0 && (
            <div>
               <div className={classes.title}>Previous revisions</div>
               <StatusCeil name={'oldReady'} value={oldReady} title={'Ready'} />
               <StatusCeil name={'oldInProgress'} value={oldInProgress} title={'In progress'} />
               <StatusCeil name={'oldFailed'} value={oldFailed} title={'Failed'} />
            </div>
         )}
         {unknownCount > 0 && (
            <div>
               <div className={classes.title}>Unknown state</div>
               <StatusCeil name={'unavailable'} value={unavailable} title={'Unavailable (YP is unavailable)'} />
               <StatusCeil name={'unknown'} value={unknown} title={'Unknown'} />
            </div>
         )}
      </div>
   );
});

ReplicaSetStatuses.displayName = 'ReplicaSetStatuses';
