import { faCheck } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Tabs, TabsItemProps } from '@yandex-cloud/uikit';
import React, { useState } from 'react';

import { LostClusterWarningPlate } from '../../../../../../../../components/common';
import { LocationApprovalStatus, ReplicaSetInfo, ReplicaSetType } from '../../../../../../../../models/ui';
import { getReplicaSetLocationInfo } from '../../../../../../../../models/ui/status';
import { useStage } from '../../../../../../../../redux/hooks';
import { useDeployUnitStatusView, useReplicaSetStatusView } from '../../../../hooks';
import { ReplicaSetApproves } from '../ReplicaSetApproves/ReplicaSetApproves';
import { ReplicaSetDisruptionBudget } from '../ReplicaSetDisruptionBudget/ReplicaSetDisruptionBudget';
import { ReplicaSetEndpointSets } from '../ReplicaSetEndpointSets/ReplicaSetEndpointSets';
import { ReplicaSetPods } from '../ReplicaSetPods/ReplicaSetPods';
import { ReplicaSetStatuses } from '../ReplicaSetStatuses/ReplicaSetStatuses';

import classes from './ReplicaSetTabs.module.css';

enum ReplicaSetTab {
   Pods = 'pods',
   Status = 'status',
   Endpoints = 'endpoints',
   DisruptionBudget = 'disruptionBudget',
   Approves = 'approves',
}

const tabList: TabsItemProps[] = [
   {
      id: ReplicaSetTab.Pods,
      title: (
         <span className={classes.tab} data-test={'StatusPage:Pods'}>
            Pods
         </span>
      ),
   },
   {
      id: ReplicaSetTab.Status,
      title: (
         <span className={classes.tab} data-test={'StatusPage:Status'}>
            Status
         </span>
      ),
   },
   {
      id: ReplicaSetTab.Endpoints,
      title: (
         <span className={classes.tab} data-test={'StatusPage:Endpoints'}>
            Endpoints
         </span>
      ),
   },
   {
      id: ReplicaSetTab.DisruptionBudget,
      title: (
         <span className={classes.tab} data-test={'StatusPage:DisruptionBudget'}>
            Disruption budget
         </span>
      ),
   },
];

const approveTab: (existLocationApprove: boolean) => TabsItemProps = existLocationApprove => ({
   id: ReplicaSetTab.Approves,
   title: (
      <span className={classes.tab} data-test={'StatusPage:Approves'}>
         Approves {existLocationApprove && <FontAwesomeIcon icon={faCheck} color={'var(--deploy-color-success)'} />}
      </span>
   ),
});

interface Props {
   replicaSetInfo: ReplicaSetInfo;
   stageId: string;
   duId: string;
}

export const ReplicaSetTabs: React.FC<Props> = React.memo(({ replicaSetInfo, stageId, duId }) => {
   const { id, locations, type, endpointSets } = replicaSetInfo;

   const { stage } = useStage(stageId);

   const [activeTab, setActiveTab] = useState(ReplicaSetTab.Pods);

   const { replicaSetLocationInfo } = getReplicaSetLocationInfo({ replicaSetData: replicaSetInfo });
   const { ypLocation } = replicaSetLocationInfo;

   const { replicaSetStatusViewList } = useReplicaSetStatusView({
      replicaSetSelectData: [{ id, location: ypLocation }],
   });
   const [replicaSetStatusView = null] = replicaSetStatusViewList;

   const { deployUnitStatusViewList } = useDeployUnitStatusView({ deployUnitSelectData: [{ stageId, duId }] });
   const [deployUnitStatusView = null] = deployUnitStatusViewList;

   if (!deployUnitStatusView) {
      return null;
   }

   /** only for `ReplicaSetType.PerCluster` */
   const oneLocation = type === ReplicaSetType.PerCluster ? Array.from(locations.values())[0] : null;
   const isDisabled =
      oneLocation && stage?.labels?.deploy?.disabled_clusters?.includes([...replicaSetInfo.locations][0]);

   const { spec, overrides, locationApproves } = deployUnitStatusView;

   const { needApproval } = {
      ...spec.perLocationSettings,
      ...(overrides?.deploySettings?.perLocationSettings ?? {}),
   };

   const needLocationApprove = type === ReplicaSetType.PerCluster && needApproval.has(oneLocation!);
   const existLocationApprove =
      type === ReplicaSetType.PerCluster &&
      locationApproves.get(oneLocation!)?.status === LocationApprovalStatus.Approved;

   const tabItems: TabsItemProps[] = [...tabList, ...(needLocationApprove ? [approveTab(existLocationApprove)] : [])];

   const noDataMessage = `No data about pods in ${oneLocation?.toUpperCase() ?? 'this'} location`;
   const noPodsMessage = `No pods in ${oneLocation?.toUpperCase() ?? 'this'} location`;

   const totalPods = replicaSetStatusView?.podsSummary.total;

   return (
      <div className={classes.tabs}>
         {isDisabled && oneLocation && <LostClusterWarningPlate cluster={oneLocation} isDisabled={isDisabled} />}
         {
            <>
               <Tabs
                  onSelectTab={tabId => setActiveTab(tabId as ReplicaSetTab)}
                  activeTab={activeTab}
                  items={tabItems}
               />
               <div className={classes.content}>
                  {{
                     [ReplicaSetTab.Pods]: () =>
                        totalPods !== undefined && totalPods !== 0 ? (
                           <ReplicaSetPods stageId={stageId} duId={duId} replicaSetInfo={replicaSetInfo} />
                        ) : (
                           <div className={classes.noData}>
                              {totalPods === undefined ? noDataMessage : noPodsMessage}
                           </div>
                        ),
                     [ReplicaSetTab.Status]: () =>
                        replicaSetStatusView?.podsSummary?.total ? (
                           <ReplicaSetStatuses podsSummary={replicaSetStatusView?.podsSummary ?? null} />
                        ) : (
                           <div className={classes.noData}>
                              {totalPods === undefined ? noDataMessage : noPodsMessage}
                           </div>
                        ),
                     [ReplicaSetTab.Endpoints]: () => <ReplicaSetEndpointSets endpointSets={endpointSets} />,
                     [ReplicaSetTab.DisruptionBudget]: () => (
                        <ReplicaSetDisruptionBudget
                           replicaSetInfo={replicaSetInfo}
                           deployUnit={deployUnitStatusView}
                           stageId={stageId}
                           podsTotal={replicaSetStatusView?.podsSummary?.total} // undefined = unknown
                           disruptionBudgets={replicaSetStatusView?.replicaSetDisruptionBudgets ?? null}
                        />
                     ),
                     [ReplicaSetTab.Approves]: () => (
                        <ReplicaSetApproves
                           needApproval={needLocationApprove}
                           stageId={stageId}
                           duId={duId}
                           location={oneLocation!}
                        />
                     ),
                  }[activeTab]()}
               </div>
            </>
         }
      </div>
   );
});

ReplicaSetTabs.displayName = 'ReplicaSetTabs';
