import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { useHistory, useLocation } from 'react-router';
import { Spin } from '@yandex-cloud/uikit';
import { EmptyContainer, EmptyContainerType, toQuery } from '@yandex-infracloud-ui/libs';

import { useStage, useStageProblems } from '../../../../../../redux';
import {
   getInitialProblemFilters,
   parseFiltersFromUrl,
   ProblemFilters,
   isInitialFilters,
   StageProblemFilters,
} from '../StageProblemFilters/StageProblemFilters';
import { StageProblemList } from '../StageProblemList/StageProblemList';

interface Props {
   stageId: string;
}
export const StageHealth: React.FC<Props> = React.memo(({ stageId }) => {
   const { stage } = useStage(stageId);
   const duIds = (stage?.deployUnits ?? []).map(du => du.id);
   const history = useHistory();
   const location = useLocation();

   const { issues, isLoading } = useStageProblems(stageId);
   const [filters, setFilters] = useState(getInitialProblemFilters);
   const initialFilters = useMemo(() => isInitialFilters(filters), [filters]);

   const updateFilters = useCallback(
      (value: ProblemFilters) => {
         history.push({ search: toQuery(value) });
      },
      [history],
   );

   useEffect(() => {
      const filterFromUrl = parseFiltersFromUrl(location.search);
      setFilters(filterFromUrl);
   }, [location.search]);

   if (isLoading) {
      return <Spin />;
   }

   const list = issues.filter(issue => {
      const duPass = filters.deployUnitIds.size === 0 || filters.deployUnitIds.has(issue.locations.deployUnitId ?? '');
      const levelPass = filters.level.size === 0 || filters.level.has(issue.level);
      const classPass = filters.class.size === 0 || filters.class.has(issue.class);
      return duPass && levelPass && classPass;
   });

   return (
      <>
         <StageProblemFilters filters={filters} updateFilters={updateFilters} availableDeployUnits={duIds} />
         {list.length === 0 ? (
            <EmptyContainer
               type={EmptyContainerType.EmptyState}
               title={'No problems'}
               description={initialFilters ? 'Stage has great health!' : 'Try to change the filters'}
            />
         ) : (
            <StageProblemList values={list} stageId={stageId} />
         )}
      </>
   );
});

StageHealth.displayName = 'StageHealth';
