import React from 'react';
import { YCSelect } from '@yandex-data-ui/common';
import { fromQuery, queryValueToSet } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';

import {
   CommonIssueFilters,
   getInitialIssueFilters,
   IssueClass,
   IssueFilters,
   IssueLevel,
} from '../../../../../../modules/infra-doctor';

import classes from './StageProblemFilters.module.css';

export interface ProblemFilters extends CommonIssueFilters {
   deployUnitIds: Set<string>;
}

export const getInitialProblemFilters: () => ProblemFilters = () => ({
   ...getInitialIssueFilters(),
   deployUnitIds: new Set(),
});

export function isInitialFilters(filters: ProblemFilters): boolean {
   const { deployUnitIds, class: issueClass, level } = filters;
   return deployUnitIds.size === 0 && issueClass.size === 0 && level.size === 0;
}

type UrlFilters = {
   [k in keyof ProblemFilters]: string | undefined;
};

export function parseFiltersFromUrl(search: string): ProblemFilters {
   const parsed = fromQuery(search) as Partial<UrlFilters>;

   return {
      deployUnitIds: queryValueToSet(parsed.deployUnitIds),
      class: queryValueToSet(parsed.class) as Set<IssueClass>,
      level: queryValueToSet(parsed.level) as Set<IssueLevel>,
   };
}

interface Props {
   filters: ProblemFilters;
   updateFilters(newFilters: ProblemFilters): void;
   availableDeployUnits: string[];
}

export const StageProblemFilters: React.FC<Props> = ({ filters, updateFilters, availableDeployUnits }) => (
   <div className={classes.filterLine}>
      <IssueFilters
         filters={filters}
         updateFilters={updateFilters}
         order={['deployUnitIds', 'level', 'class']}
         components={{
            deployUnitIds: (value, update) => (
               <YCSelect
                  type={'multiple'}
                  showSearch={false}
                  value={Array.from(value)}
                  items={availableDeployUnits.map(du => ({ value: du, title: du }))}
                  onUpdate={ids => update(new Set(ids))}
                  placeholder={'Deploy unit'}
               />
            ),
         }}
      />
      <Button onClick={() => updateFilters(getInitialProblemFilters())}>Reset</Button>
   </div>
);

StageProblemFilters.displayName = 'StageProblemFilters';
