import React from 'react';

import { useHistory } from 'react-router';
import { Button, HelpPopover } from '@yandex-cloud/uikit';

import { RichMessage } from '../../../../../../components';
import { urlBuilder } from '../../../../../../models';
import { IssueActionCode, IssueTable } from '../../../../../../modules/infra-doctor';
import { StageIssue } from '../../../../../../models/ui';

const actionTitles: Record<IssueActionCode, string> = {
   'update-runtime-version': 'Update runtime revision',
   'update-sidecars': 'Update sidecars',
   'help': 'Edit in form',
   '': 'Edit in form',
};
interface Props {
   stageId: string;
   values: StageIssue[];
}

export const StageProblemList: React.FC<Props> = ({ values, stageId }) => {
   const history = useHistory();
   return (
      <>
         <IssueTable
            columns={new Set(['description', 'level', 'class'])}
            customColumns={new Set(['deployUnit', 'action'] as const)}
            order={['description', 'level', 'class', 'deployUnit', 'action']}
            data={values}
            columnSettings={{
               deployUnit: {
                  title: 'Deploy unit',
                  sort: true,
                  renderItem: issue => issue.locations.deployUnitId ?? '',
               },
               action: {
                  title: '',
                  renderItem: issue => {
                     const { action, locations } = issue;
                     const { code = '', message } = action;

                     const { deployUnitId, boxId, workloadId, isResource } = locations;

                     const editNodeLink = urlBuilder.stageEdit(stageId, {
                        duId: deployUnitId ?? undefined,
                        boxId: deployUnitId && boxId ? boxId : undefined,
                        workloadId: deployUnitId && boxId && workloadId ? workloadId : undefined,
                     });
                     const editLink = deployUnitId && isResource ? `${editNodeLink}#disks` : editNodeLink;

                     const links: Record<IssueActionCode, string> = {
                        'update-runtime-version': urlBuilder.stageEdit(stageId, { duId: deployUnitId ?? undefined }),
                        'update-sidecars': urlBuilder.stageUpdate(stageId),
                        'help': editLink,
                        '': editLink,
                     };

                     return (
                        <div>
                           <Button view={'normal'} size={'m'} onClick={() => history.push(links[code])}>
                              {actionTitles[code]}
                           </Button>
                           {message && <HelpPopover content={<RichMessage text={message} />} />}
                        </div>
                     );
                  },
               },
            }}
         />
      </>
   );
};

StageProblemList.displayName = 'StageProblemList';
