import React, { ReactNode } from 'react';

import { faCheckCircle, faExclamationCircle } from '@fortawesome/pro-regular-svg-icons';
import { faExclamationTriangle } from '@fortawesome/pro-solid-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Label, LabelProps } from '@yandex-cloud/uikit';
import { ExternalLink, Loader } from '@yandex-infracloud-ui/libs';

import { XRayStatusSummary, XRaySummaryType } from '../../../../../../models/ui/stage/XRay';
import { EXTERNAL_LINKS } from '../../../../../../models';

import classes from './StageRevisionXRayStatus.module.css';

const xRayThemes: Record<XRaySummaryType, LabelProps['theme']> = {
   ok: 'success',
   critical: 'danger',
   info: 'info',
   warning: 'warning',
   loading: 'unknown',
   failed: 'unknown',
   unknown: 'unknown',
};

const xRayClasses: Record<XRaySummaryType, string> = {
   ok: classes.ok,
   critical: classes.danger,
   info: classes.info,
   warning: classes.warning,
   loading: classes.default,
   failed: classes.default,
   unknown: classes.default,
};

const xRayTitles: Partial<Record<XRaySummaryType, string>> = {
   failed: 'X-Ray failed',
};

const xRayIcons: Record<XRaySummaryType, ReactNode> = {
   ok: <FontAwesomeIcon icon={faCheckCircle} />,
   critical: <FontAwesomeIcon icon={faExclamationCircle} />,
   info: <FontAwesomeIcon icon={faExclamationCircle} />,
   warning: <FontAwesomeIcon icon={faExclamationCircle} />,
   loading: <Loader />,
   failed: <FontAwesomeIcon icon={faExclamationTriangle} />,
   unknown: <strong>?</strong>,
};

const xRayDescriptions: Record<XRaySummaryType, string> = {
   ok: 'Congratulations! No issues found. You rock!',
   critical: 'Critical warning',
   info: 'Info warning',
   warning: 'Warning',
   loading: 'X-Ray analysis is not completed yet',
   failed: 'Request for X-Ray analysis failed',
   unknown: 'Unknown',
};

interface Props {
   href: string;
   xRayStatus: XRayStatusSummary;
}

export const StageRevisionXRayStatus: React.FC<Props> = ({ href, xRayStatus }: Props) => {
   const { issues, stageHealth, analysisStatus, summaryType } = xRayStatus;

   const XrayLabel = (
      <Label theme={xRayThemes[summaryType]} className={xRayClasses[summaryType]} icon={xRayIcons[summaryType]}>
         {xRayTitles[summaryType] ?? 'X-Ray'}
      </Label>
   );

   const XrayInfo = (
      <div className={classes.tooltipContentWrapper}>
         <div className={classes.summaryLine}>
            <div className={classes.label}>{XrayLabel}</div>
            {analysisStatus === 'done' && stageHealth !== 'ok' ? (
               <div className={classes.warningsMessage}>
                  <div>Issues:</div>
                  {issues.high > 0 && <div className={classes.danger}>{issues.high} high</div>}
                  {issues.medium > 0 && <div className={classes.warning}>{issues.medium} medium</div>}
                  {issues.low > 0 && <div className={classes.info}>{issues.low} low</div>}
               </div>
            ) : (
               <div>{xRayDescriptions[summaryType]}</div>
            )}
         </div>
         <div>
            Security and vulnerability{' '}
            <ExternalLink href={href} className={classes.xRayLink}>
               analysis of this stage
            </ExternalLink>
            <ExternalLink href={EXTERNAL_LINKS.wiki.xRay}>Read documentation</ExternalLink>
         </div>
      </div>
   );

   return <div style={{ padding: '8px 0' }}>{XrayInfo}</div>;
};
StageRevisionXRayStatus.displayName = 'StageRevisionXRayStatus';
