import { useCallback, useEffect, useMemo } from 'react';

import { useSelector } from 'react-redux';

import { TMultiClusterReplicaSet, TPod, TReplicaSet } from '../../../../../proto-typings';
import { RootState, selectDuPods, selectReplicaSets, useDeployUnit, useStage } from '../../../../../redux';
import {
   DeployUnit,
   PodConverter,
   ReplicaSetConverter,
   Stage,
   StageStatus,
   StageStatusLabel,
   ReplicaSet,
   DeployUnitStatus,
} from '../../../../../models/ui';
import {
   DeployUnitStatusView,
   getDeployUnitStatusView,
   getReplicaSetStatusView,
   getStageStatusView,
   PodStatusView,
   ReplicaSetStatusView,
   StageStatusView,
   StatusTtl,
   getPodStatusView,
} from '../../../../../models/ui/status';
import { createKey, getEmptyStoreAction, restoreObjectFromKey, StoreAction, updateStore } from '../../../../../utils';
import { StatusViewStore, StatusViewStoreUpdater } from '../context';

type Props = {
   /**
    * текущий стейдж
    */
   stageId: string;
   /**
    * текущий деплой-юнит
    */
   duId: string;
   /**
    * ключ текущего реплика-сета = id + location
    */
   replicaSetKey: string;
   /**
    * выбранные кластеры - в общем случае не совпадает с location (например xdc для MCRS может включать несколько кластеров sas/vla/...)
    */
   activeClusters: Set<string>;

   statusViewStore: StatusViewStore;

   updateStatusViewStore: StatusViewStoreUpdater;
};

export function useStatusViewUpdate({
   stageId,
   duId,
   replicaSetKey,
   activeClusters,
   statusViewStore,
   updateStatusViewStore,
}: Props): void {
   // redux часть

   // берём нужные объекты из redux, обновление ссылок на них также служит триггером к обновлению дерева,
   // так как ссылки из redux меняются только при загрузке новых данных из апи

   // данные текущего стейджа и его деплой-юнитов
   const { stage, stageStatus, stageStatusLabel } = useStage(stageId);

   // данные текущего деплой-юнита
   const { deployUnit, deployUnitStatus } = useDeployUnit(stageId, duId);

   // данные реплика-сетов
   // список ключей реплика-сетов
   const { replicaSetKeys, replicaSetKeyData } = useMemo(() => getReplicaSetDataFromDu({ deployUnitStatus }), [
      deployUnitStatus,
   ]);
   const replicaSetSelector = useCallback((state: RootState) => selectReplicaSets(state, replicaSetKeyData), [
      replicaSetKeyData,
   ]);

   const rawReplicaSets = useSelector(replicaSetSelector) as (TReplicaSet | TMultiClusterReplicaSet | undefined)[];
   const replicaSetList = useMemo(
      () =>
         rawReplicaSets.map((rawReplicaSet, i) => {
            const { location } = replicaSetKeyData[i];
            return rawReplicaSet ? ReplicaSetConverter.getUIModel(rawReplicaSet, location) : null;
         }),
      [rawReplicaSets, replicaSetKeyData],
   );
   const replicaSets: Record<string, ReplicaSet | null> = Object.fromEntries(
      replicaSetKeys.map((key, i) => [key, replicaSetList[i]]),
   );

   // данные подов
   const podSetId = PodConverter.getPodSetId(stageId, duId);

   const podSelector = useMemo(() => createPodSelector(podSetId), [podSetId]);
   const rawPods = useSelector(podSelector);

   // обновление снизу вверх по дереву
   const store = statusViewStore;

   // обновление подов
   const { podsStoreUpdateAction, podsView } = useMemo(() => getPodStoreData({ rawPods, deployUnit, activeClusters }), [
      activeClusters,
      deployUnit,
      rawPods,
   ]);

   // обновление реплика-сетов
   const { replicaSetStoreUpdateAction, podStoreClearAction, replicaSetStatusViewList } = useMemo(
      () =>
         getReplicaSetStoreData({
            store,
            replicaSetKeys,
            replicaSets,
            activePodsView: podsView,
            activeReplicaSetKey: replicaSetKey,
            deployUnit,
            deployUnitStatus,
            activeClusters,
            stage,
         }),
      [
         activeClusters,
         deployUnit,
         deployUnitStatus,
         podsView,
         replicaSetKey,
         replicaSetKeys,
         replicaSets,
         stage,
         store,
      ],
   );

   // обновление деплой-юнитов
   const { deployUnitStoreUpdateAction, deployUnitStatusViewList, replicaSetStoreClearAction } = useMemo(
      () =>
         getDeployUnitStoreData({
            store,
            stage,
            stageId,
            stageStatus,
            activeDuId: duId,
            activeReplicaSetStatusViewList: replicaSetStatusViewList,
         }),
      [duId, replicaSetStatusViewList, stage, stageId, stageStatus, store],
   );

   // обновление стейджа
   const { stageStoreUpdateAction } = useMemo(
      () =>
         getStageStoreData({
            stageId,
            stage,
            stageStatus,
            stageStatusLabel,
            activeDeployUnitStatusViewList: deployUnitStatusViewList,
         }),
      [deployUnitStatusViewList, stage, stageId, stageStatus, stageStatusLabel],
   );

   // обновление контекста
   useEffect(() => {
      updateStatusViewStore(oldStatusViewStore => {
         const podsStore = updateStore(oldStatusViewStore.pods, podsStoreUpdateAction);
         const podsStoreNew = updateStore(podsStore, podStoreClearAction);

         const rsStore = updateStore(oldStatusViewStore.replicaSets, replicaSetStoreUpdateAction);
         const rsStoreNew = updateStore(rsStore, replicaSetStoreClearAction);

         const duStore = updateStore(oldStatusViewStore.deployUnits, deployUnitStoreUpdateAction);
         const stageStore = updateStore(oldStatusViewStore.stages, stageStoreUpdateAction);

         const podKeysByReplicaSetKey = getPodKeysByReplicaSetKey({ replicaSetList, rawPods });

         return {
            pods: podsStoreNew,
            replicaSets: rsStoreNew,
            deployUnits: duStore,
            stages: stageStore,

            podKeysByReplicaSetKey,
         };
      });

      // вручную выставляем зависимости из redux и входных данных
      // eslint-disable-next-line react-hooks/exhaustive-deps
   }, [replicaSetList, rawPods, stageStatus, stageId, duId, replicaSetKey, activeClusters]);
}

function getPodKeysByReplicaSetKey({
   replicaSetList,
   rawPods,
}: {
   replicaSetList: (ReplicaSet | null)[];
   rawPods: PodRecord[];
}): Record<string, string[]> {
   const podKeysByReplicaSetKey: Record<string, string[]> = {};

   for (const replicaSet of replicaSetList) {
      if (replicaSet) {
         const rsKey = createKey({ id: replicaSet.id, location: replicaSet.location });
         podKeysByReplicaSetKey[rsKey] = rawPods
            .filter(record => replicaSet.locations.has(record.meta.cluster))
            .map(record => createKey({ id: record.pod.meta?.id ?? '', location: record.meta.cluster }));
      }
   }

   return podKeysByReplicaSetKey;
}

type PodRecord = {
   pod: TPod;
   meta: { cluster: string };
};

function createPodSelector(podSetId: string) {
   return (state: RootState): PodRecord[] =>
      Object.entries(selectDuPods(state, podSetId)).flatMap(([cluster, pods]) =>
         pods.map(pod => ({ pod: pod as TPod, meta: { cluster } })),
      );
}

function getReplicaSetDataFromDu({
   deployUnitStatus,
}: {
   deployUnitStatus: DeployUnitStatus | null;
}): { replicaSetKeys: string[]; replicaSetKeyData: { location: string; id: string }[] } {
   const replicaSetKeys = Object.keys(deployUnitStatus?.replicaSetsInfo ?? {});
   const replicaSetKeyData = replicaSetKeys.map(key => restoreObjectFromKey(key) as { location: string; id: string });

   return { replicaSetKeys, replicaSetKeyData };
}

function getPodStoreData({
   rawPods,
   deployUnit,
   activeClusters,
}: {
   rawPods: PodRecord[];
   deployUnit: DeployUnit | null;
   activeClusters: Set<string>;
}): { podsStoreUpdateAction: StoreAction<PodStatusView>; podsView: PodStatusView[] } {
   const podsView = rawPods
      .filter(({ meta }) => activeClusters.has(meta.cluster))
      .map(({ pod, meta }) => getPodStatusView({ pod: PodConverter.getPod(pod), deployUnit, location: meta.cluster }));

   const podsStoreUpdateAction = getEmptyStoreAction<PodStatusView>();

   podsStoreUpdateAction.write = Object.fromEntries(
      podsView.map(podView => [createKey({ id: podView.id, location: podView.location }), { value: podView }]),
   );

   return { podsStoreUpdateAction, podsView };
}

function getReplicaSetStoreData({
   store,
   replicaSetKeys,
   replicaSets,
   activeReplicaSetKey,
   activePodsView,
   deployUnit,
   deployUnitStatus,
   activeClusters,
   stage,
}: {
   store: StatusViewStore;
   replicaSetKeys: string[];
   activeReplicaSetKey: string;
   activePodsView: PodStatusView[];
   replicaSets: Record<string, ReplicaSet | null>;
   deployUnit: DeployUnit | null;
   deployUnitStatus: DeployUnitStatus | null;
   activeClusters: Set<string>;
   stage: Stage | null;
}): {
   replicaSetStoreUpdateAction: StoreAction<ReplicaSetStatusView>;
   podStoreClearAction: StoreAction<PodStatusView>;
   replicaSetStatusViewList: ReplicaSetStatusView[];
} {
   const replicaSetStoreUpdateAction = getEmptyStoreAction<ReplicaSetStatusView>();
   const podStoreClearAction = getEmptyStoreAction<PodStatusView>();
   const replicaSetStatusViewList: ReplicaSetStatusView[] = [];
   const rsStore = store.replicaSets;

   const now = Date.now();

   for (const key of replicaSetKeys) {
      const currentReplicaSet: ReplicaSet | null = replicaSets[key] ?? null;
      if (!currentReplicaSet) {
         // eslint-disable-next-line no-continue
         continue;
      }
      let podViewList: PodStatusView[] = [];

      // поды в кэше
      const podKeyList = store.podKeysByReplicaSetKey[key] ?? [];
      const cachePods = podKeyList.map(podKey => store.pods[podKey]).filter(Boolean);

      if (key === activeReplicaSetKey) {
         // активный реплика-сет

         // сначала определим видимые поды для реплика-сета
         const visiblePods = activePodsView.filter(view => currentReplicaSet.locations.has(view.location));
         const visiblePodIdSet = new Set(visiblePods.map(view => view.id));

         const notVisibleCachePods = cachePods.filter(
            view => !activeClusters.has(view.location) && !visiblePodIdSet.has(view.id),
         );

         // невидимые поды, но недавно обновленные, используем
         const notVisiblePods = notVisibleCachePods.filter(view => now - view.lastUpdateTimestamp < StatusTtl.pod);

         // поды для расчёта реплика-сета
         podViewList = [...visiblePods, ...notVisiblePods];

         // невидимые поды, но давно не обновляемые, их стираем
         const oldNotVisiblePods = notVisibleCachePods.filter(view => now - view.lastUpdateTimestamp >= StatusTtl.pod);
         podStoreClearAction.delete.push(...oldNotVisiblePods.map(({ id, location }) => createKey({ id, location })));
      } else if (rsStore.hasOwnProperty(key)) {
         // был посчитан ранее
         const oldRsView = rsStore[key];
         const { lastUpdateTimestamp } = oldRsView;
         const lifeTime = now - lastUpdateTimestamp;

         if (lifeTime < StatusTtl.replicaSet) {
            // относительно свежие используют кэш

            // свежие поды
            const notVisiblePods = cachePods.filter(view => now - view.lastUpdateTimestamp < StatusTtl.pod);
            podViewList = notVisiblePods;

            // устаревшие поды
            const oldNotVisiblePods = cachePods.filter(view => now - view.lastUpdateTimestamp >= StatusTtl.pod);
            podStoreClearAction.delete.push(...oldNotVisiblePods.map(view => view.id));
         } else {
            // старые остаются с пустым списком подов и стирают поды из кэша
            podStoreClearAction.delete.push(...podKeyList);
         }
      }

      const replicaSetStatusView = getReplicaSetStatusView({
         replicaSet: currentReplicaSet,
         visiblePods: podViewList,
         deployUnit,
         deployUnitStatus,
         stage,
      });

      replicaSetStoreUpdateAction.write[key] = { value: replicaSetStatusView };
      replicaSetStatusViewList.push(replicaSetStatusView);
   }

   return { replicaSetStoreUpdateAction, replicaSetStatusViewList, podStoreClearAction };
}

function getDeployUnitStoreData({
   store,
   stageId,
   stage,
   stageStatus,
   activeDuId,
   activeReplicaSetStatusViewList,
}: {
   store: StatusViewStore;
   stageId: string;
   activeDuId: string;
   stageStatus: StageStatus | null;
   stage: Stage | null;
   activeReplicaSetStatusViewList: ReplicaSetStatusView[];
}): {
   deployUnitStoreUpdateAction: StoreAction<DeployUnitStatusView>;
   replicaSetStoreClearAction: StoreAction<ReplicaSetStatusView>;
   deployUnitStatusViewList: DeployUnitStatusView[];
} {
   const deployUnitStoreUpdateAction = getEmptyStoreAction<DeployUnitStatusView>();
   const replicaSetStoreClearAction = getEmptyStoreAction<ReplicaSetStatusView>();
   const deployUnitStatusViewList: DeployUnitStatusView[] = [];

   const deployUnitIds = Array.from(stageStatus?.deployUnits?.keys() ?? []);

   const duStore = store.deployUnits;
   const rsStore = store.replicaSets;

   const now = Date.now();
   const deployUnitMap = new Map((stage?.deployUnits ?? []).map(du => [du.id, du]));

   for (const currentDuId of deployUnitIds) {
      const currentDeployUnitStatus = stageStatus?.deployUnits?.get(currentDuId) ?? null;
      const currentDeployUnit = deployUnitMap.get(currentDuId) ?? null;
      if (!currentDeployUnitStatus || !currentDeployUnit) {
         // eslint-disable-next-line no-continue
         continue;
      }
      const duKey = createKey({ stageId, duId: currentDuId });

      const { replicaSetKeys } = getReplicaSetDataFromDu({ deployUnitStatus: currentDeployUnitStatus });

      let replicaSetData: ReplicaSetStatusView[] = [];
      if (currentDuId === activeDuId) {
         // активный деплой-юнит
         replicaSetData = activeReplicaSetStatusViewList;
      } else if (duStore.hasOwnProperty(duKey)) {
         // посчитанные ранее
         const oldDuStatusView = duStore[duKey];
         const { lastUpdateTimestamp } = oldDuStatusView;
         const lifeTime = now - lastUpdateTimestamp;

         if (lifeTime < StatusTtl.deployUnit) {
            // для свежих берём rs из кэша
            replicaSetData = replicaSetKeys.map(key => rsStore[key]).filter(Boolean);
         } else {
            // для старых оставляем пустой список rs и стираем из кэша существующие
            replicaSetStoreClearAction.delete.push(...replicaSetKeys);
         }
      }

      const { deployUnitStatusView } = getDeployUnitStatusView({
         deployUnit: currentDeployUnit,
         deployUnitStatus: currentDeployUnitStatus,
         replicaSetData,
      });

      deployUnitStoreUpdateAction.write[duKey] = { value: deployUnitStatusView };
      deployUnitStatusViewList.push(deployUnitStatusView);
   }

   return { deployUnitStoreUpdateAction, replicaSetStoreClearAction, deployUnitStatusViewList };
}

function getStageStoreData({
   stageId,
   stage,
   stageStatus,
   stageStatusLabel,
   activeDeployUnitStatusViewList,
}: {
   stageId: string;
   stageStatus: StageStatus | null;
   stage: Stage | null;
   stageStatusLabel: StageStatusLabel | null;
   activeDeployUnitStatusViewList: DeployUnitStatusView[];
}): {
   stageStoreUpdateAction: StoreAction<StageStatusView>;
} {
   const stageStoreUpdateAction = getEmptyStoreAction<StageStatusView>();

   if (stage && stageStatus && stageStatusLabel) {
      const { stageStatusView } = getStageStatusView({
         stage,
         stageStatus,
         stageStatusLabel,
         deployUnitData: activeDeployUnitStatusViewList,
      });

      stageStoreUpdateAction.write[createKey({ stageId })] = { value: stageStatusView };
   }

   return { stageStoreUpdateAction };
}
