import React, { ReactNode } from 'react';
import { IconProp } from '@fortawesome/fontawesome-svg-core';
import { EPodCurrentState } from '../../../../proto-typings';
import { StatusState } from '../../../../components/lib';
import {
   PodCurrentState,
   PodCustomCurrentState,
   PodObjectName,
   PodObjectType,
   ypViewConfig,
} from '../../../../models/ui';
import { BoxStatus } from './components/BoxStatus/BoxStatus';
import { WorkloadStatus } from './components/WorkloadStatus/WorkloadStatus';
import { LayerStatus } from './components/LayerStatus/LayerStatus';
import { StaticResourceStatus } from './components/StaticResourceStatus/StaticResourceStatus';
import { VolumeStatus } from './components/VolumeStatus/VolumeStatus';
import { Storage } from '../../../../models/Storage';

/**
 * группы статусов, назначаются для удобства
 */
export type PodStatusGroup = 'any' | 'ready' | 'notready' | 'inprogress' | 'failed';

export type PodRevisionGroup = 'any' | 'current' | 'previous';

export interface PodFilters {
   statusGroup: PodStatusGroup;
   revisionGroup: PodRevisionGroup;
   isCustomFilter: boolean;
   customFilter: string;
}

export const defaultPodFilters: PodFilters = {
   statusGroup: 'any',
   revisionGroup: 'any',
   isCustomFilter: false,
   customFilter: '',
};

export const podStatusGroupData: Record<PodStatusGroup, { value: string; title: string }> = {
   any: {
      value: '',
      title: 'any status',
   },
   ready: {
      value: '[/status/agent/pod_agent_payload/status/ready/status]="true"',
      title: 'status is ready',
   },
   notready: {
      value: 'NOT [/status/agent/pod_agent_payload/status/ready/status]="true"',
      title: 'status is not ready',
   },
   inprogress: {
      value: '[/status/agent/pod_agent_payload/status/in_progress/status]="true"',
      title: 'status is in progress',
   },
   failed: {
      title: 'status is failed',
      value: '[/status/agent/pod_agent_payload/status/failed/status]="true"',
   },
};

export const podStatusGroupList: PodStatusGroup[] = ['any', 'ready', 'notready', 'inprogress', 'failed'];

export const podRevisionGroupData: (
   revsion: number,
) => Record<PodRevisionGroup, { value: string; title: string }> = revision => ({
   any: {
      value: '',
      title: 'any revision',
   },
   current: {
      value: `[/status/agent/pod_agent_payload/status/revision]=${revision}u`,
      title: `current revision: ${revision}`,
   },
   previous: {
      value: `NOT [/status/agent/pod_agent_payload/status/revision]=${revision}u`,
      title: `previous revisions`,
   },
});

export const podRevisionGroupList: PodRevisionGroup[] = ['any', 'current', 'previous'];

export const podsPerPages = [10, 20, 50, 100, 500];

export interface PodPagination {
   limit: number;
   page: number;
}

export const defaultPodPagination: PodPagination = {
   limit: podsPerPages[0],
   page: 1,
};

export const statusMap: Record<PodCurrentState, StatusState> = {
   [EPodCurrentState.PCS_STARTED]: StatusState.Ok,
   [EPodCurrentState.PCS_START_FAILED]: StatusState.Error,
   [EPodCurrentState.PCS_START_PENDING]: StatusState.Progress,
   [EPodCurrentState.PCS_STOPPED]: StatusState.Inactive,
   [EPodCurrentState.PCS_STOP_PENDING]: StatusState.Progress,
   [EPodCurrentState.PCS_UNKNOWN]: StatusState.Unknown,
   [PodCustomCurrentState.PCS_SUSPECTED]: StatusState.Unknown,
};

export function getFetchQuery(filters: PodFilters, revision: number): string {
   const { isCustomFilter, customFilter } = filters;

   if (isCustomFilter) {
      return customFilter;
   }

   // @nikolaichev предлагается к удалению из window.localStorage, сейчас игнорируются
   // pods-filter-status
   // pods-filter-revision

   return getQueryFromMainFilters(filters, revision);
}

export function getQueryFromMainFilters(filters: PodFilters, revision: number): string {
   const { statusGroup, revisionGroup } = filters;
   return [podStatusGroupData[statusGroup].value, podRevisionGroupData(revision)[revisionGroup].value]
      .filter(Boolean)
      .map(filter => `(${filter})`)
      .join(' AND ');
}

export function getFiltersFromUrl(search: URLSearchParams): PodFilters {
   const filters = defaultPodFilters;
   const query = search.get('query');
   const statusGroup = search.get('status');
   const revisionGroup = search.get('revision');

   if (query) {
      filters.isCustomFilter = true;
      filters.customFilter = query;
   }
   if (statusGroup && getStatusGroup(statusGroup)) {
      filters.statusGroup = statusGroup;
   }
   if (revisionGroup && getRevisionGroup(revisionGroup)) {
      filters.revisionGroup = revisionGroup;
   }

   return filters;
}

function getStatusGroup(value: string): value is PodStatusGroup {
   return podStatusGroupData.hasOwnProperty(value);
}

function getRevisionGroup(value: string): value is PodRevisionGroup {
   return podRevisionGroupData(0).hasOwnProperty(value);
}

export const podsPaginationPerPageKey = 'pods-pagination-per-page';
export const statusViewSettingsVisible = 'status-view-settings-visible';

export const statusSettingsStorage = new Storage();

export function getPaginationFromUrl(search: URLSearchParams): PodPagination {
   const pagination = { ...defaultPodPagination };

   const limit = search.get('per_page');

   // независим от урла, меняется только явно
   const storageLimit = statusSettingsStorage.getItem<number>(podsPaginationPerPageKey, defaultPodPagination.limit);

   const page = search.get('page');

   if (limit && podsPerPages.includes(Number(limit))) {
      pagination.limit = Number(limit);
   } else if (storageLimit && podsPerPages.includes(Number(storageLimit))) {
      pagination.limit = storageLimit;
   }

   const pageN = parseInt(page ?? '', 10);
   if (page && !isNaN(pageN)) {
      pagination.page = pageN;
   }

   return pagination;
}

type UrlKey = 'query' | 'status' | 'revision' | 'page' | 'per_page';

export function getUrlParamsFromPodData(
   filters: PodFilters,
   pagination: PodPagination,
): Partial<Record<UrlKey, string | null>> {
   const params: ReturnType<typeof getUrlParamsFromPodData> = {};

   const { statusGroup, revisionGroup, isCustomFilter, customFilter } = filters;

   if (isCustomFilter && customFilter) {
      params.query = customFilter;
   }

   if (statusGroup !== defaultPodFilters.statusGroup) {
      params.status = statusGroup;
   }

   if (revisionGroup !== defaultPodFilters.revisionGroup) {
      params.revision = revisionGroup;
   }

   const { limit, page } = pagination;

   if (limit !== defaultPodPagination.limit) {
      params.per_page = String(limit);
   }

   if (page !== defaultPodPagination.page) {
      params.page = String(page);
   }

   return params;
}

interface PodObjectData {
   icon: IconProp;
   status: ReactNode;
   inlineStatus: ReactNode;
   isBoxChild: boolean;
}

export const PodObjectDataConfig: {
   [name in PodObjectName]: (object: PodObjectType[name]) => PodObjectData;
} = {
   [PodObjectName.Box]: box => ({
      icon: ypViewConfig.box.fontAwesomeIcon,
      status: <BoxStatus box={box} />,
      inlineStatus: <BoxStatus box={box} inline={true} />,
      isBoxChild: false,
   }),
   [PodObjectName.Workload]: workload => ({
      icon: ypViewConfig.workload.fontAwesomeIcon,
      status: <WorkloadStatus workload={workload} />,
      inlineStatus: <WorkloadStatus workload={workload} inline={true} />,
      isBoxChild: true,
   }),
   [PodObjectName.Layer]: layer => ({
      icon: ypViewConfig.layer.fontAwesomeIcon,
      status: <LayerStatus layer={layer} />,
      inlineStatus: <LayerStatus layer={layer} inline={true} />,
      isBoxChild: true,
   }),
   [PodObjectName.StaticResource]: resource => ({
      icon: ypViewConfig.staticResource.fontAwesomeIcon,
      status: <StaticResourceStatus resource={resource} />,
      inlineStatus: <StaticResourceStatus resource={resource} inline={true} />,
      isBoxChild: true,
   }),
   [PodObjectName.Volume]: volume => ({
      icon: ypViewConfig.volume.fontAwesomeIcon,
      status: <VolumeStatus volume={volume} />,
      inlineStatus: <VolumeStatus volume={volume} inline={true} />,
      isBoxChild: false,
   }),
};
