import React, { useCallback, useMemo, useState } from 'react';
import { Popover } from '@yandex-cloud/uikit';

import type { SchemasSettings } from 'monaco-yaml';
import { useTitle } from '@yandex-infracloud-ui/libs';
import { RouteComponentProps } from 'react-router';
import yaml from 'js-yaml';
import {
   getPrettyMessageText,
   MonacoEditorProps,
   MonacoEditorValidationInfo,
   YamlEditor,
} from '@yandex-infracloud-ui/monaco-editor';

import { useStage } from '../../../../redux';

import classes from './StageYamlPage.module.css';

import { useJSONSchemaByPath } from '../../../../json-schema';
import { getDeployJSONUri } from '../../../../json-schema/model';

export const stageSpecJSONSchema: SchemasSettings = {
   fileMatch: ['*'],
   schema: {
      type: 'object',
      properties: {
         revision: {
            description: 'ревизия стейджа',
            type: 'number',
            minimum: 0,
         },
         deploy_units: {
            type: 'object',
            patternProperties: {
               '.*': {
                  type: 'object',
                  properties: {
                     network_defaults: {
                        type: 'object',
                        properties: {
                           network_id: {
                              type: 'string',
                              description: 'Сеть в контейнерах',
                           },
                        },
                     },
                  },
               },
            },
         },
      },
   },
   uri: 'https://deploy.yandex-team.ru/yp/stageSpec.json',
};

const stageSpecPath = '/yp/TStageSpec';
const stageSpecUri = getDeployJSONUri(stageSpecPath);

export const StageYamlPage: React.FC<RouteComponentProps<{ stageId: string }>> = React.memo(({ match }) => {
   const { stageId } = match.params;
   const { rawStage } = useStage(stageId, true);
   const [value, setValue] = useState(() => yaml.dump(rawStage?.spec ?? ''));
   const [validationInfo, setValidationInfo] = useState<MonacoEditorValidationInfo | null>(null);
   const onChangeValidate: Required<MonacoEditorProps>['onChangeValidate'] = useCallback(({ validationInfo: info }) => {
      setValidationInfo(info);
   }, []);

   const prettyMessages = useMemo(() => validationInfo?.messages.map(message => getPrettyMessageText({ message })), [
      validationInfo?.messages,
   ]);

   useTitle(`YAML / ${stageId}`);

   useJSONSchemaByPath(stageSpecPath);

   return rawStage ? (
      <div className={classes.wrapper}>
         <h2>
            Stage spec for {stageId}
            {validationInfo?.summary.summaryStatus === 'error' && (
               <Popover
                  content={
                     <p>
                        {prettyMessages?.map(message => (
                           <span>{message}</span>
                        ))}
                     </p>
                  }
               >
                  <span style={{ color: 'red', marginLeft: '1rem', fontSize: '1rem' }}>errors</span>
               </Popover>
            )}
         </h2>
         <YamlEditor value={value} onUpdate={setValue} onChangeValidate={onChangeValidate} uri={stageSpecUri} />
      </div>
   ) : null;
});

StageYamlPage.displayName = 'StageYamlPage';
