import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { Loader, Button } from '@yandex-cloud/uikit';
import { useSelector } from 'react-redux';
import { RouteComponentProps, useHistory } from 'react-router';

import { PageTemplate } from '../../../../components/yp';
import { YpLocation } from '../../../../models/api';
import { ReplicaSetFiltersParams } from '../../../../models/ui/yp/models';
import { getYPFilter, RootState } from '../../../../redux';
import { setUrlQuery } from '../../../../utils';
import { getMulticlusterReplicaSetQueryFromFilter } from '../../../../utils/yp';
import { getNavigation, parseFiltersValuesFromSearch } from '../../utils';
import { ReplicaSetFilters } from '../../../../components/yp/ReplicaSetFilters/ReplicaSetFilters';
import { selectYpMulticlusterReplicaSets } from '../../../../redux/slices';
import { ReplicaSetsTable } from '../../../../components/yp/ReplicaSetsTable/ReplicaSetsTable';
import { useYpMulticlusterReplicaSets } from '../../../../redux/hooks';
import { YpEntityRouteProps } from '../../../../models/ui/yp/view';

import classes from './MulticlusterReplicaSets.module.css';

const REQUEST_KEY = 'ypMulticlusterReplicaSetsRequest';

const defaultFilters: ReplicaSetFiltersParams = {};

export const MulticlusterReplicaSets: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({ location }) => {
   const navigation = useMemo(
      () => getNavigation('xdc', { id: 'multicluster-replica-sets', title: 'multicluster replica sets' }),
      [],
   );
   const { requestReplicaSets, isFetching } = useYpMulticlusterReplicaSets(REQUEST_KEY);

   const history = useHistory();

   const [filters, setFilters] = useState<ReplicaSetFiltersParams>(defaultFilters);
   const filterExpression = useMemo(() => getMulticlusterReplicaSetQueryFromFilter(filters), [filters]);
   const replicaSetsFilterSelector = useCallback(
      (state: RootState) => getYPFilter(state, YpLocation.XDC, 'multiclusterReplicaSets', filterExpression),
      [filterExpression],
   );
   const { ids: replicaSetIds, canLoadMore = false } = useSelector(replicaSetsFilterSelector) ?? {};
   const replicaSetsSelector = useCallback(
      (state: RootState) => selectYpMulticlusterReplicaSets(state, replicaSetIds),
      [replicaSetIds],
   );
   const replicaSets = useSelector(replicaSetsSelector);

   useEffect(() => {
      if (!location.search) {
         setUrlQuery(history, location, defaultFilters, false);
         setFilters(defaultFilters);
         requestReplicaSets(filters, true);
      } else {
         const query = parseFiltersValuesFromSearch(location.search) as ReplicaSetFiltersParams;
         setFilters(query);
         requestReplicaSets(query, true);
      }
   }, []); //eslint-disable-line

   const handleFilters = (newFilters: ReplicaSetFiltersParams) => {
      setUrlQuery(history, location, newFilters, false);
      setFilters(newFilters);
      requestReplicaSets(newFilters, true);
   };

   const onLoadMore = useCallback(() => {
      requestReplicaSets(filters, false);
   }, [filters, requestReplicaSets]);

   return (
      <PageTemplate error={undefined} navigation={navigation}>
         <div>
            <ReplicaSetFilters onSubmit={handleFilters} filters={filters} disabled={false} />
         </div>
         {replicaSets && (
            <div>
               <ReplicaSetsTable replicaSets={replicaSets} cluster={YpLocation.XDC} type={'TMulticlusterReplicaSet'} />
            </div>
         )}
         <div className={classes.loaderContainer}>
            {isFetching && <Loader size={'s'} />}
            {canLoadMore && !isFetching && (
               <Button onClick={onLoadMore} view={'outlined'}>
                  Load more
               </Button>
            )}
         </div>
      </PageTemplate>
   );
};
