import React, { useMemo } from 'react';

import { Json } from '@yandex-infracloud-ui/libs';
import { RouteComponentProps } from 'react-router';
import { Loader } from '@yandex-cloud/uikit';

import { PageTemplate, Tab, Tabs } from '../../../../../components/yp';
import { YpLocation } from '../../../../../models/api';
import { useNetworkErrors } from '../../../../../redux';
import { getNavigation, getYPError } from '../../../utils';
import { ReplicaSetInfo } from '../../../../../components/yp/ReplicaSetInfo/ReplicaSetInfo';
import { useYpMulticlusterReplicaSet } from '../../../../../redux/hooks';
import { YpEntityRouteProps } from '../../../../../models/ui/yp/view';

import classes from './MulticlusterReplicaSet.module.css';

const networkRequests = {
   replicaSet: 'ypMulticlusterReplicaSetRequest',
   pods: 'ypMulticlusterReplicaSetPodsRequest',
};

export const MulticlusterReplicaSet: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { entityId },
   },
}) => {
   const navigation = useMemo(() => getNavigation(YpLocation.XDC, 'multicluster-replica-sets', entityId), [entityId]);
   const { replicaSet } = useYpMulticlusterReplicaSet(entityId, networkRequests.replicaSet);

   const { meta, labels, spec, status } = replicaSet ?? {};

   const errors = useNetworkErrors(Object.values(networkRequests));

   const replicaSetError = useMemo(() => getYPError(errors[networkRequests.replicaSet]), [errors]);

   if (!replicaSet && !Object.values(errors).length) {
      return (
         <div className={classes.loaderContainer}>
            <span className={classes.loader}>
               <Loader size={'s'} />
            </span>
            <span className={classes.loaderGap} />
         </div>
      );
   }

   const tabs: Tab[] = ([
      replicaSet && {
         id: 'info',
         title: 'General',
         content: (
            <div>
               <ReplicaSetInfo replicaSet={replicaSet} cluster={YpLocation.XDC} type={'TMultiClusterReplicaSet'} />
            </div>
         ),
      },
      meta && {
         id: 'meta',
         title: 'Meta',
         content: <Json obj={meta} />,
      },
      labels && {
         id: 'labels',
         title: 'Labels',
         content: <Json obj={labels} />,
      },
      spec && {
         id: 'spec',
         title: 'Spec',
         content: <Json obj={spec} />,
      },
      status && {
         id: 'status',
         title: 'Status',
         content: <Json obj={status} />,
      },
   ] as (Tab | undefined)[]).filter((item): item is Tab => Boolean(item));

   return (
      <PageTemplate error={replicaSetError} navigation={navigation}>
         <Tabs tabs={tabs} />
      </PageTemplate>
   );
};
