import { Button, TextInput } from '@yandex-cloud/uikit';
import { YCSelect, YCSelectItem } from '@yandex-data-ui/common';
import { classNames, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { NodeFiltersParams } from '../../../../../models/ui/yp/view';
import { EHfsmState, ENodeMaintenanceState, TNodeSegment } from '../../../../../proto-typings';

import classes from './NodeFilters.module.css';

const stateSelectorItems = Object.values(EHfsmState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

const maintenanceSelectorItems = Object.values(ENodeMaintenanceState).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

interface Props {
   filters: NodeFiltersParams;
   segments: Pick<TNodeSegment, 'meta'>[];
   disabled: boolean;
   className?: string;

   onSubmit(params: NodeFiltersParams): void;
}

export const NodeFilters: React.FC<Props> = ({ onSubmit, filters, segments = [], disabled, className }) => {
   const [filtersState, setFiltersState] = useState<NodeFiltersParams>(filters);

   const segmentItems: YCSelectItem[] = useMemo(() => {
      const items: (YCSelectItem | null)[] = segments.map(segment => {
         if (segment.meta?.id) {
            return { value: segment.meta.id, title: segment.meta.id };
         }
         return null;
      });

      return items.filter((item): item is YCSelectItem => Boolean(item));
   }, [segments]);

   useEffect(() => {
      setFiltersState(filters);
   }, [filters]);

   const onInputChange = useCallback(
      <T extends keyof NodeFiltersParams>(fieldName: T, fieldValue: Required<NodeFiltersParams[T]>) => {
         const newFiltersState = { ...filtersState, [fieldName]: fieldValue };
         setFiltersState(newFiltersState);
      },
      [filtersState],
   );

   return (
      <div className={classNames(classes.filters, className)}>
         <div className={classes.row}>
            <TextInput
               value={filtersState.nodeId ?? ''}
               onUpdate={value => {
                  onInputChange('nodeId', value);
               }}
               placeholder={'Node ID'}
               disabled={disabled}
            />
            <YCSelect
               label={'Segment: '}
               type={YCSelect.MULTIPLE}
               items={segmentItems}
               value={filtersState.segments}
               onUpdate={value => {
                  onInputChange('segments', value);
               }}
               disabled={disabled}
            />
            <YCSelect
               label={'State: '}
               type={YCSelect.MULTIPLE}
               items={stateSelectorItems}
               value={filtersState.state}
               onUpdate={value => {
                  onInputChange('state', value as EHfsmState[]);
               }}
               disabled={disabled}
            />
            <YCSelect
               label={'Maintenance: '}
               type={YCSelect.MULTIPLE}
               items={maintenanceSelectorItems}
               value={filtersState.maintenance}
               onUpdate={value => {
                  onInputChange('maintenance', value as ENodeMaintenanceState[]);
               }}
               disabled={disabled}
            />
            <Button
               onClick={() => {
                  onSubmit({ segments: ['default'] });
               }}
               disabled={disabled}
            >
               Reset
            </Button>
            <Button
               onClick={() => {
                  onSubmit({ ...filtersState });
               }}
               disabled={disabled}
            >
               Search
            </Button>
         </div>
         <div className={classes.row}>
            <TextInput
               value={filtersState.query}
               onUpdate={value => {
                  onInputChange('query', value);
               }}
               placeholder={'Query example: [/labels/cpu_flags/avx2] = %true'}
               disabled={disabled}
            />
         </div>
      </div>
   );
};
