import React, { ReactNode, useEffect, useMemo, useState } from 'react';

import { Button, Loader, Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { RouteComponentProps, useHistory } from 'react-router';
import { Link } from 'react-router-dom';

import { Status, StatusState } from '../../../../components/lib';
import { FormatDate, PageTemplate } from '../../../../components/yp';
import { urlBuilder } from '../../../../models';
import { YpLocation } from '../../../../models/api';
import { NodeMaintenanceStateMap, NodeStateMap } from '../../../../models/ui/yp/models';
import { NodeFiltersParams, YpEntityRouteProps } from '../../../../models/ui/yp/view';
import { useNetworkErrors, useYpNodes, useYpNodesSegments } from '../../../../redux';
import { setUrlQuery } from '../../../../utils';
import { getNavigation, getYPError } from '../../utils';
import { NodeFilters } from './NodeFilters/NodeFilters';
import { parseFiltersValuesFromSearch } from './utils';

import classes from './Nodes.module.css';

interface TableRowItem {
   nodeId: ReactNode;
   segment: string;
   state: ReactNode;
   lastSeenTime: ReactNode;
   maintenanceState: ReactNode;
}

const baseColumns: TableColumnConfig<TableRowItem>[] = [
   {
      id: 'nodeId',
      name: 'Node',
      sticky: 'left',
   },
   {
      id: 'segment',
      name: 'Segment',
      width: 250,
   },
   {
      id: 'lastSeenTime',
      name: 'Last seen time',
      width: 250,
   },
   {
      id: 'state',
      name: 'State',
      width: 250,
   },
   {
      id: 'maintenanceState',
      name: 'Maintenance state',
      width: 250,
   },
];

const networkRequests = {
   ypNode: 'ypNodesRequestKey',
   ypNodeSegment: 'ypNodesSegmentsRequestKey',
};

const defaultFilters: NodeFiltersParams = {
   nodeId: '',
   state: [],
   maintenance: [],
   segments: ['default'],
   query: '',
};

export const Nodes: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { cluster },
   },
   location,
}) => {
   const navigation = useMemo(() => getNavigation(cluster, 'nodes'), [cluster]);
   const history = useHistory();

   const [filters, setFilters] = useState<NodeFiltersParams>({});
   useEffect(() => {
      if (!location.search) {
         setUrlQuery(history, location, defaultFilters, false);
         setFilters(defaultFilters);
         requestNodes(defaultFilters, true);
      } else {
         const query = parseFiltersValuesFromSearch(location.search) as NodeFiltersParams;
         setFilters(query);
         requestNodes(query, true);
      }
   }, []); //eslint-disable-line

   const handleFilters = (newFilters: NodeFiltersParams) => {
      setUrlQuery(history, location, newFilters, false);
      setFilters(newFilters);
      requestNodes(newFilters, true);
   };

   const { nodes, isFetching, canFetch, requestNodes } = useYpNodes(cluster as YpLocation, networkRequests.ypNode);
   const segments = useYpNodesSegments(cluster as YpLocation, networkRequests.ypNodeSegment);

   const errors = useNetworkErrors(Object.values(networkRequests));
   const errorItems = useMemo(
      () =>
         Object.values(networkRequests)
            .filter(Boolean)
            .map(errorKey => getYPError(errors[errorKey], classes.error)),
      [errors],
   );

   const rows = nodes.map(node => {
      const nodeId = node?.meta?.id ?? '';
      const nodeState = node.status?.hfsm?.state;
      const stateStr = nodeState ? forHumanCapitalized(nodeState.toString().replace(/_/g, ' ')) : undefined;
      const state = nodeState ? NodeStateMap.get(nodeState) : undefined;
      const maintenanceState = node?.status?.maintenance?.state ?? '';

      const segment = node.labels?.segment ?? '';

      return {
         nodeId: <Link to={urlBuilder.ypNode(cluster, nodeId)}>{nodeId}</Link>,
         segment,
         state: (
            <Status state={state || StatusState.Error} isAnimated={false}>
               {stateStr}
            </Status>
         ),
         lastSeenTime: <FormatDate timestamp={node.status?.last_seen_time} empty={'Unknown'} />,
         maintenanceState: maintenanceState && (
            <Status state={NodeMaintenanceStateMap.get(maintenanceState) || StatusState.Unknown} isAnimated={false}>
               {forHumanCapitalized(maintenanceState)}
            </Status>
         ),
      };
   });

   return (
      <PageTemplate navigation={navigation} error={errorItems.length ? errorItems : undefined}>
         <div className={classes.contentContainer}>
            <NodeFilters disabled={isFetching} filters={filters} segments={segments} onSubmit={handleFilters} />
            <Table
               columns={baseColumns}
               data={rows}
               className={classes.tableContainer}
               emptyMessage={isFetching ? 'Loading nodes...' : undefined}
            />
            <div className={classes.loaderContainer}>
               {isFetching && <Loader size={'s'} />}
               {canFetch && !isFetching && (
                  <Button
                     onClick={() => {
                        requestNodes(filters, false);
                     }}
                     view={'outlined'}
                  >
                     Load more
                  </Button>
               )}
            </div>
         </div>
      </PageTemplate>
   );
};
