import React, { useMemo } from 'react';

import { Loader } from '@yandex-cloud/uikit';
import { RouteComponentProps } from 'react-router-dom';

import { YpLocation } from '../../../../../models/api';
import { useNetworkErrors, useYpNode, useYpNodePods, useYpNodeResources } from '../../../../../redux';
import { getNavigation, getYPError } from '../../../utils';
import { nodeErrorKeys } from './models';
import { NodeInfo } from './NodeInfo/NodeInfo';
import { NodeResources } from './NodeResources/NodeResources';
import { NodeHistory } from './NodeHistory/NodeHistory';
import { PageTemplate, PodsTable, Tab, Tabs } from '../../../../../components/yp';
import { YpEntityRouteProps } from '../../../../../models/ui/yp/view';

import classes from './Node.module.css';

export const Node: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({ match }) => {
   const {
      params: { cluster, entityId },
   } = match;

   const node = useYpNode(entityId, cluster as YpLocation, 'ypNodeRequestKey');
   const resources = useYpNodeResources(entityId, cluster as YpLocation, 'ypNodeResourcesRequestKey');
   const pods = useYpNodePods(entityId, cluster as YpLocation, 'ypNodePodsRequestKey');
   const navigation = useMemo(() => getNavigation(cluster, 'nodes', entityId), [cluster, entityId]);
   const nodeErrors = useNetworkErrors([nodeErrorKeys.node, nodeErrorKeys.resources]);
   const nodeErrorItems = useMemo(
      () =>
         Object.values(nodeErrors)
            .filter(Boolean)
            .map(error => getYPError(error, classes.error)),
      [nodeErrors],
   );
   const podErrors = useNetworkErrors([nodeErrorKeys.pods]);
   const podsErrorItem = useMemo(() => getYPError(podErrors[nodeErrorKeys.pods]), [podErrors]);

   if (!node && !nodeErrorItems.length) {
      return (
         <div className={classes.loaderContainer}>
            <span className={classes.loader}>
               <Loader size={'s'} />
            </span>
            <span className={classes.loaderGap} />
         </div>
      );
   }

   const showPods = (pods || podsErrorItem) && resources;
   const tabs: Tab[] = node
      ? ([
           {
              id: 'info',
              title: 'General',
              content: (
                 <div className={classes.infoTab}>
                    <div className={classes.generalInfo}>
                       <NodeInfo node={node} />
                    </div>

                    {resources ? (
                       <div className={classes.resources}>
                          <NodeResources resources={resources} />
                       </div>
                    ) : null}
                 </div>
              ),
           },
           { id: 'meta', title: 'Meta', obj: node.meta },
           { id: 'labels', title: 'Labels', obj: node.labels },
           { id: 'spec', title: 'Spec', obj: node.spec },
           { id: 'status', title: 'Status', obj: node.status },
           { id: 'resources', title: 'Resources', obj: resources },
           {
              id: 'history',
              title: 'History',
              content: (
                 <NodeHistory
                    nodeId={entityId}
                    uuid={node?.meta?.uuid}
                    cluster={cluster as YpLocation}
                    requestKey={'ypNodeHistoryRequestKey'}
                 />
              ),
           },
           showPods && {
              id: 'pods',
              title: <span>Pods {pods && <span className={classes.podsCount}>{pods.length}</span>}</span>,
              content: (
                 <div>
                    {podsErrorItem ? (
                       <div className={classes.podsError}>{podsErrorItem}</div>
                    ) : (
                       <PodsTable
                          pods={pods}
                          nodeResources={resources}
                          cluster={cluster as YpLocation}
                          emptyText={'No pods found'}
                       />
                    )}
                 </div>
              ),
              width: 'full',
           },
        ] as (Tab | undefined)[]).filter((item): item is Tab => Boolean(item))
      : [];

   return (
      <PageTemplate navigation={navigation} error={nodeErrorItems.length ? nodeErrorItems : undefined}>
         {!node ? null : <Tabs tabs={tabs} />}
      </PageTemplate>
   );
};

Node.displayName = 'Node';
