import React from 'react';

import { CopyableText, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { formatDistance } from 'date-fns';

import { StatusState } from '../../../../../../components/lib';
import { EEventType, TNodeStatus_THfsm, TNodeStatus_TMaintenance } from '../../../../../../proto-typings';
import { TimelineEvent, TimelineRow } from '../../../../../../components/yp';
import { NodeMaintenanceStateMap, NodeStateMap } from '../../../../../../models/ui/yp/models';
import { isNodeHistoryItemsEqual, NodeHistoryEvent } from '../../../../../../models/ui/yp/view';

const getHFSMViewData = (item: TNodeStatus_THfsm | null, prevItem: TNodeStatus_THfsm | null, eventType: EEventType) => {
   const isEqual = isNodeHistoryItemsEqual(item, prevItem);
   let state;
   if (eventType === EEventType.ET_OBJECT_REMOVED) {
      state = StatusState.Inactive;
   } else if (item) {
      state = NodeStateMap.get(item.state) ?? StatusState.Unknown;
   } else {
      state = StatusState.Inactive;
   }

   const title = eventType === EEventType.ET_OBJECT_REMOVED ? 'Pod removed' : forHumanCapitalized(item?.state ?? '');
   const isChanged = eventType === EEventType.ET_OBJECT_REMOVED ? true : !isEqual;

   return {
      isChanged,
      title,
      state,
      message: item?.message,
   };
};

const getMaintenanceViewData = (
   item: TNodeStatus_TMaintenance | null,
   prevItem: TNodeStatus_TMaintenance | null,
   eventType: EEventType,
): TimelineEvent => {
   const isEqual = isNodeHistoryItemsEqual(item, prevItem);
   let state;
   if (eventType === EEventType.ET_OBJECT_REMOVED) {
      state = StatusState.Inactive;
   } else if (item) {
      state = NodeMaintenanceStateMap.get(item.state) ?? StatusState.Unknown;
   } else {
      state = StatusState.Inactive;
   }

   const content = [];
   if (item?.info) {
      const { disruptive, estimated_duration, id, kind, node_set_id, message, priority, source, uuid } = item.info;

      content.push(
         {
            title: 'Info',
            content: <span />,
            key: 'info',
         },
         {
            title: 'ID:',
            content: id && (
               <span>
                  {id}
                  <CopyableText text={id} />
               </span>
            ),
            key: 'id',
         },
         {
            title: 'UUID:',
            content: uuid && (
               <span>
                  {uuid}
                  <CopyableText text={uuid} />
               </span>
            ),
            key: 'uuid',
         },
         {
            title: 'Kind:',
            content: kind && forHumanCapitalized(kind),
            key: 'kind',
         },
         {
            title: 'Node set ID:',
            content: node_set_id && (
               <span>
                  {node_set_id}
                  <CopyableText text={node_set_id} />
               </span>
            ),
            key: 'node-set-id',
         },
         {
            title: 'Message:',
            content: message,
            key: 'message',
         },
         {
            title: 'Disruptive:',
            content: disruptive.toString(),
            key: 'disruptive',
         },
         {
            title: 'Estimated duration:',
            content:
               estimated_duration?.seconds && formatDistance(new Date(estimated_duration.seconds * 1000), new Date(0)),
            key: 'duration',
         },
         {
            title: 'Priority:',
            content: priority && forHumanCapitalized(priority),
            key: 'proprity',
         },
         {
            title: 'Source:',
            content: source && forHumanCapitalized(source),
            key: 'source',
         },
      );
   }

   const title = eventType === EEventType.ET_OBJECT_REMOVED ? 'Pod removed' : forHumanCapitalized(item?.state ?? '');
   const isChanged = eventType === EEventType.ET_OBJECT_REMOVED ? true : !isEqual;

   return {
      isChanged,
      title,
      state,
      message: item?.message,
      content: content.length ? content : undefined,
   };
};

export const getHistoryRows = (history: NodeHistoryEvent[], cluster: string): TimelineRow[] =>
   history.map((item, index) => {
      const {
         result: {
            status: { hfsm, maintenance },
         },
         event_type,
         time: { seconds },
         user,
      } = item;

      const statusViewData = getHFSMViewData(hfsm, history[index + 1]?.result.status.hfsm, event_type);
      const maintenanceViewData = getMaintenanceViewData(
         maintenance,
         history[index + 1]?.result.status.maintenance,
         event_type,
      );

      return {
         timestamp: seconds,
         events: [statusViewData, maintenanceViewData],
         user,
      };
   });
