import React from 'react';

import { ExternalLink, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import { formatDuration } from 'date-fns';

import { DefinitionList, DefinitionListItem, Status, StatusState } from '../../../../../../components/lib';
import { EXTERNAL_LINKS } from '../../../../../../models';
import { EHfsmState, ENodeMaintenanceState, TNode } from '../../../../../../proto-typings';
import { FormatDate } from '../../../../../../components/yp/FormatDate/FormatDate';
import { NodeMaintenanceStateMap, NodeStateMap } from '../../../../../../models/ui/yp/models';
import { InfoGroup, Labels } from '../../../../../../components/yp';

import classes from './NodeInfo.module.css';

const cpuLabels = ['avx', 'avx2'];

interface Props {
   node: TNode;
}

export const NodeInfo: React.FC<Props> = ({ node }) => {
   const nodeId = node.meta?.id ?? '';

   const nodeState = node.status?.hfsm?.state;
   const stateStr = nodeState ? forHumanCapitalized(nodeState.toString().replace(/_/g, ' ')) : undefined;
   const status = nodeState ? NodeStateMap.get(nodeState) : undefined;

   const hfsmMessage = nodeState !== EHfsmState.HS_UP ? node.status?.hfsm?.message : undefined;

   const labels = cpuLabels
      .filter(label => (node.labels ? node.labels[label] : false))
      .map(label => (
         <span key={label} className={classes.label}>
            {label}
         </span>
      ));
   if (node.labels?.extras?.network?.bandwidth_10G) {
      labels.push(
         <span key={'bandwidth_10G'} className={classes.label}>
            network bandwidth 10G
         </span>,
      );
   }

   const links = [
      <ExternalLink key={'wall-e'} href={EXTERNAL_LINKS.walleNode(nodeId)}>
         Wall-E
      </ExternalLink>,
      <ExternalLink key={'golovan'} href={EXTERNAL_LINKS.yasmPanel('Host', { hosts: nodeId }, false)}>
         YASM
      </ExternalLink>,
   ];

   const fields: DefinitionListItem[] = [
      {
         name: 'Node ID',
         content: nodeId,
         copyText: true,
      },
      {
         name: 'Segment',
         content: node.labels?.segment,
      },
      {
         name: 'Data center',
         content: node.labels?.topology?.dc,
      },
      {
         name: 'Rack',
         content: node.labels?.topology?.rack,
      },
      {
         name: 'State',
         content: (
            <Status state={status || StatusState.Error} isAnimated={false}>
               {stateStr}
            </Status>
         ),
      },
      {
         name: 'State update time',
         content: <FormatDate timestamp={node?.status?.hfsm?.last_updated} empty={'Unknown'} />,
      },
      {
         name: 'Message',
         content: hfsmMessage,
      },
      {
         name: 'Last seen time',
         content: <FormatDate timestamp={node.status?.last_seen_time} empty={'Unknown'} />,
      },
      {
         name: 'Labels',
         content: labels.length ? <div className={classes.labels}>{labels}</div> : null,
      },
      {
         name: 'Links',
         content: links,
      },
      {
         name: 'CPU Model',
         content: node?.labels?.cpu_model,
      },
   ];

   const maintenanceFields: DefinitionListItem[] = [];
   const maintenance = node?.status?.maintenance;
   if (maintenance?.state && maintenance?.state !== ENodeMaintenanceState.NMS_NONE) {
      maintenanceFields.push(
         {
            name: 'ID',
            content: maintenance.info?.id,
            copyText: true,
         },
         {
            name: 'State',
            content: (
               <Status
                  state={NodeMaintenanceStateMap.get(maintenance?.state) || StatusState.Unknown}
                  isAnimated={false}
               >
                  {forHumanCapitalized(maintenance?.state)}
               </Status>
            ),
         },
         {
            name: 'Update time',
            content: <FormatDate timestamp={maintenance.last_updated} empty={'Unknown'} />,
         },
         {
            name: 'Status message',
            content: maintenance.status_message,
         },
         {
            name: 'Disruptive',
            content: maintenance.info?.disruptive,
         },
         {
            name: 'Estimated duration',
            content: maintenance.info?.estimated_duration ? formatDuration(maintenance.info.estimated_duration) : null,
         },
         {
            name: 'Kind',
            content: maintenance.info?.kind ? forHumanCapitalized(maintenance.info?.kind) : null,
         },
         {
            name: 'Message',
            content: maintenance.info?.message,
         },
         {
            name: 'Node set id',
            content: maintenance.info?.node_set_id,
            copyText: true,
         },
         {
            name: 'Priority',
            content: maintenance.info?.priority ? forHumanCapitalized(maintenance.info?.priority) : null,
         },
         {
            name: 'Source',
            content: maintenance.info?.source,
         },
         {
            name: 'UUID',
            content: maintenance.info?.uuid,
            copyText: true,
         },
      );
   }

   const alerts = node?.status?.alerts ?? [];
   const alertsContent = alerts.map(alertData => (
      <InfoGroup>
         <InfoGroup margin={'row'}>
            <DefinitionList
               items={[
                  {
                     name: 'UUID',
                     content: alertData.uuid,
                     copyText: true,
                  },
                  {
                     content: alertData.type,
                     name: 'Type',
                  },
                  {
                     name: 'Creation time',
                     content: <FormatDate timestamp={alertData.creation_time} />,
                  },
                  {
                     name: 'Description',
                     content: alertData.description,
                  },
               ]}
            />
         </InfoGroup>
         <Labels labels={alertData.labels} />
      </InfoGroup>
   ));

   return (
      <>
         <InfoGroup>
            <DefinitionList items={fields} />
         </InfoGroup>
         {maintenanceFields.length ? (
            <InfoGroup title={'Maintenance'}>
               <DefinitionList items={maintenanceFields} />
               <Labels labels={maintenance?.info?.labels} />
            </InfoGroup>
         ) : null}
         {alertsContent.length ? <InfoGroup title={'Alerts'}>{alertsContent}</InfoGroup> : null}
      </>
   );
};

NodeInfo.displayName = 'Node__NodeInfo';
