import { classNames, formatNumber } from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useMemo } from 'react';
import { ResourceBar } from '../../../../../../components/yp';
import {
   CPUResource,
   DiskResource,
   getResourcesByKind,
   GPUResource,
   Resource,
} from '../../../../../../models/ui/yp/models';
import { EResourceKind, TResource } from '../../../../../../proto-typings';
import { formatBytes, formatCores } from '../../../../../../utils/yp';

import classes from './NodeResources.module.css';

interface Props {
   resources: TResource[];
}

const createResourceRows = (resources: (Resource | undefined)[]): ReactNode[] =>
   resources.filter(Boolean).map((resource, index) => {
      if (!resource) {
         return null;
      }

      if (resource.kind === EResourceKind.RK_CPU) {
         const cpuResource = resource as CPUResource;

         return (
            <ResourceBar
               key={cpuResource.id}
               title={`CPU (cpu to vcpu factor: ${cpuResource.cpuToVcpuFactor})`}
               total={cpuResource.total / 1000}
               used={cpuResource.used / 1000}
               free={cpuResource.free / 1000}
               format={formatCores}
               suffix={' vcores'}
               className={classes.resourceBar}
            />
         );
      }

      if (resource.kind === EResourceKind.RK_DISK) {
         const diskResource = resource as DiskResource;

         return [
            <div
               className={classNames(classes.diskTitle, classes.resourceR)}
            >{`Disk (${diskResource.storageClass} class, ${diskResource.storageProvisioner} provisioner)`}</div>,
            <ResourceBar
               key={`${diskResource.id}_capacity`}
               title={'Capacity'}
               total={diskResource.total}
               used={diskResource.used}
               free={diskResource.free}
               format={formatBytes}
               className={classes.diskBar}
            />,
            <ResourceBar
               key={`${diskResource.id}_bandwidth`}
               title={'Bandwidth'}
               total={diskResource.bandwidthTotal}
               used={diskResource.bandwidthUsed}
               free={diskResource.bandwidthFree}
               format={formatBytes}
               suffix={'/s'}
               className={classes.diskBar}
            />,
            <ResourceBar
               key={`${diskResource.id}_disk_slot`}
               title={'Disk slot'}
               total={diskResource.slotsTotal}
               used={diskResource.slotsUsed}
               free={diskResource.slotsFree}
               className={classes.resourceBar}
            />,
         ];
      }

      if (resource.kind === EResourceKind.RK_GPU) {
         const gpuResource = resource as GPUResource;

         return (
            <ResourceBar
               key={gpuResource.id}
               title={`GPU (${gpuResource.model.replace(/_/g, ' ')} ${formatNumber(gpuResource.memory, formatBytes)})`}
               total={gpuResource.total}
               used={gpuResource.used}
               free={gpuResource.total - gpuResource.used}
               className={classes.resourceBar}
            />
         );
      }

      if (resource.kind === EResourceKind.RK_MEMORY) {
         return (
            <ResourceBar
               key={resource.kind}
               title={'Memory'}
               total={resource.total}
               used={resource.used}
               free={resource.free}
               format={formatBytes}
               className={classes.resourceBar}
            />
         );
      }

      if (resource.kind === EResourceKind.RK_NETWORK) {
         return (
            <ResourceBar
               key={resource.id}
               title={'Network'}
               total={resource.total}
               used={resource.used}
               free={resource.free}
               format={formatBytes}
               className={classes.resourceBar}
            />
         );
      }

      if (resource.kind === EResourceKind.RK_SLOT) {
         return (
            <ResourceBar
               key={resource.id}
               title={'Slot'}
               total={resource.total}
               used={resource.used}
               free={resource.free}
               className={classes.resourceBar}
            />
         );
      }

      return null;
   });

export const NodeResources: React.FC<Props> = ({ resources }) => {
   const items = useMemo(
      () =>
         Object.values(EResourceKind)
            .map((kind: EResourceKind) => getResourcesByKind(resources, kind))
            .filter(Boolean)
            .map(arrayByKind => createResourceRows(arrayByKind as Resource[])),
      [resources],
   );

   return (
      <div className={classes.resources}>
         <div className={classes.title}>Resources</div>
         <div className={classes.items}>{items}</div>
      </div>
   );
};

NodeResources.displayName = 'Node__NodeResources';
