import { Button, TextInput } from '@yandex-cloud/uikit';
import { YCSelect, YCSelectItem } from '@yandex-data-ui/common';
import { classNames, forHumanCapitalized } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo, useState } from 'react';

import { DeployEngine, PodSetsFiltersParams } from '../../../../../models/ui/yp/view';
import { TNodeSegment } from '../../../../../proto-typings';

import classes from './PodSetFilters.module.css';

const deployEngineItems = Object.values(DeployEngine).map(key => ({
   title: forHumanCapitalized(key),
   value: key,
}));

interface Props {
   filters: PodSetsFiltersParams;
   segments: Pick<TNodeSegment, 'meta'>[];
   disabled: boolean;
   className?: string;

   onSubmit(params: PodSetsFiltersParams): void;
}

export const PodSetFilters: React.FC<Props> = ({ onSubmit, filters, segments = [], disabled, className }) => {
   const [filtersState, setFiltersState] = useState<PodSetsFiltersParams>(filters);

   const segmentItems = useMemo(
      () =>
         segments
            .map(segment => {
               if (segment.meta?.id) {
                  return { value: segment.meta.id, title: segment.meta.id };
               }
               return null;
            })
            .filter(Boolean),
      [segments],
   ) as YCSelectItem[];

   useEffect(() => {
      setFiltersState(filters);
   }, [filters]);

   const onInputChange = useCallback(
      <T extends keyof Required<PodSetsFiltersParams>>(fieldName: T, fieldValue: PodSetsFiltersParams[T]) => {
         const newFiltersState = { ...filtersState, [fieldName]: fieldValue };
         setFiltersState(newFiltersState);
      },
      [filtersState],
   );

   return (
      <div className={classNames(classes.filters, className)}>
         <div className={classes.row}>
            <TextInput
               value={filtersState.podSetId ?? ''}
               onUpdate={value => {
                  onInputChange('podSetId', value);
               }}
               placeholder={'Pod set ID'}
               disabled={disabled}
               hasClear={true}
            />
            <TextInput
               value={filtersState.serviceId ?? ''}
               onUpdate={value => {
                  onInputChange('serviceId', value);
               }}
               placeholder={'Service ID'}
               disabled={disabled}
               hasClear={true}
            />
            <YCSelect
               label={'Deploy engine: '}
               type={YCSelect.MULTIPLE}
               items={deployEngineItems}
               value={filtersState.deployEngine}
               onUpdate={value => {
                  onInputChange('deployEngine', value as DeployEngine[]);
               }}
               disabled={disabled}
            />
            <YCSelect
               label={'Segment: '}
               type={YCSelect.MULTIPLE}
               items={segmentItems}
               value={filtersState.segments}
               onUpdate={value => {
                  onInputChange('segments', value);
               }}
               disabled={disabled}
            />
            <TextInput
               value={filtersState.accountId ?? ''}
               onUpdate={value => {
                  onInputChange('accountId', value);
               }}
               placeholder={'Account ID'}
               disabled={disabled}
               hasClear={true}
            />
            <Button
               onClick={() => {
                  onSubmit({ segments: ['default'] });
               }}
               disabled={disabled}
            >
               Reset
            </Button>
            <Button
               onClick={() => {
                  onSubmit({ ...filtersState });
               }}
               disabled={disabled}
            >
               Search
            </Button>
         </div>
         <div className={classes.row}>
            <TextInput
               value={filtersState.query}
               onUpdate={value => {
                  onInputChange('query', value);
               }}
               placeholder={'Query example: [/status/scheduling/feasible] = %false'}
               disabled={disabled}
               hasClear={true}
            />
         </div>
      </div>
   );
};

PodSetFilters.displayName = 'PodSetFilters';
