import { Button, Loader } from '@yandex-cloud/uikit';
import React, { useEffect, useMemo, useState } from 'react';
import { RouteComponentProps, useHistory } from 'react-router';

import { PageTemplate, PodSetsTable } from '../../../../components/yp';
import { YpLocation } from '../../../../models/api';
import { PodSetsFiltersParams, YpEntityRouteProps } from '../../../../models/ui/yp/view';
import { useNetworkErrors, useYpNodesSegments } from '../../../../redux';
import { useYpPodSets } from '../../../../redux/hooks/useYpPodSets';
import { setUrlQuery } from '../../../../utils';
import { getNavigation, getYPError } from '../../utils';
import { PodSetFilters } from './PodSetFilters/PodSetFilters';
import { parseFiltersValuesFromSearch } from './utils';

import classes from './PodSets.module.css';

const requestKeys = {
   podSets: 'ypClusterPodSets',
   segments: 'ypClusterNodeSegments',
};

const defaultFilters: PodSetsFiltersParams = {
   segments: ['default'],
};

export const PodSets: React.FC<RouteComponentProps<YpEntityRouteProps>> = ({
   match: {
      params: { cluster },
   },
   location,
}) => {
   const { canFetch, isFetching, podSets, requestPodSets } = useYpPodSets(cluster as YpLocation, requestKeys.podSets);
   const segments = useYpNodesSegments(cluster as YpLocation, requestKeys.segments);
   const navigation = useMemo(() => getNavigation(cluster, { id: 'pod-sets', title: 'pod sets' }), [cluster]);
   const history = useHistory();

   const [filters, setFilters] = useState<PodSetsFiltersParams>({});
   useEffect(() => {
      if (!location.search) {
         setUrlQuery(history, location, defaultFilters, false);
         setFilters(defaultFilters);
         requestPodSets(defaultFilters, true);
      } else {
         const query = parseFiltersValuesFromSearch(location.search);
         setFilters(query);
         requestPodSets(query, true);
      }
   }, []); //eslint-disable-line

   const handleFilters = (newFilters: PodSetsFiltersParams) => {
      setUrlQuery(history, location, newFilters, false);
      setFilters(newFilters);
      requestPodSets(newFilters, true);
   };

   const errors = useNetworkErrors(Object.values(requestKeys));
   const errorItems = useMemo(
      () =>
         Object.values(requestKeys)
            .filter(Boolean)
            .map(errorKey => getYPError(errors[errorKey], classes.error)),
      [errors],
   );

   return (
      <PageTemplate navigation={navigation} error={errorItems.length ? errorItems : undefined}>
         <div className={classes.contentContainer}>
            <PodSetFilters disabled={isFetching} filters={filters} segments={segments} onSubmit={handleFilters} />
            <PodSetsTable cluster={cluster as YpLocation} isFetching={isFetching} podSets={podSets} />
            <div className={classes.loaderContainer}>
               {isFetching && <Loader size={'s'} />}
               {canFetch && !isFetching && (
                  <Button
                     onClick={() => {
                        requestPodSets(filters, false);
                     }}
                     view={'outlined'}
                  >
                     Load more
                  </Button>
               )}
            </div>
         </div>
      </PageTemplate>
   );
};
